/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * JBoss, Home of Professional Open Source
 * Copyright 2013 Red Hat Inc. and/or its affiliates and other contributors
 * as indicated by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.hibernate.checkstyle.filters;

import java.io.File;

import com.puppycrawl.tools.checkstyle.api.AuditEvent;
import com.puppycrawl.tools.checkstyle.api.Filter;

/**
 * Allows disabling some rules for the test suite source.
 *
 * Any violation will be suppressed if it's generated by a source file
 * having {@code /src/test/java} included in its path and if the violation
 * message contains the keyword "[not required for tests]".
 * <p>
 * A SuppressionFilter is too generic, and requires per-module configuration.
 *
 * @author Sanne Grinovero <sanne@hibernate.org>
 */
public class ExcludeTestPackages implements Filter {

	private static final String SUB_PATH = File.separator + "src" + File.separator + "test" + File.separator + "java";
	private static final String MESSAGE_DISABLE_KEYWORD = "[not required for tests]";

	@Override
	public boolean accept(AuditEvent aEvent) {
		String fileName = aEvent.getFileName();
		if ( fileName != null && fileName.contains( SUB_PATH ) ) {
			return acceptTestfileEvent( aEvent );
		}
		return true;
	}

	private boolean acceptTestfileEvent(AuditEvent aEvent) {
		String message = aEvent.getMessage();
		if ( message != null && message.contains( MESSAGE_DISABLE_KEYWORD ) ) {
			return false;
		}
		return true;
	}
}
