Internationalization
--------------------

Hammer uses [fast gettext](https://github.com/grosser/fast_gettext) for string translations. Most of the localization magic
is done for you in hammer already. If you want your plugin to support i18n it needs to register it's translation domain
at hammer's i18n module:
```ruby
require 'hammer_cli/i18n'

# namespace your i18n module
module HammerCLIAwesome
  module I18n

    # create a locale domain for your plugin
    class LocaleDomain < HammerCLI::I18n::LocaleDomain

      # lists all your source files that use _()
      def translated_files
        Dir.glob(File.join(File.dirname(__FILE__), '../**/*.rb'))
      end

      # returns path to your locale directory
      # it's typically "locale" in the root of your plugin
      def locale_dir
        File.join(File.dirname(__FILE__), '../../locale')
      end

      # name for your language domain
      def domain_name
        'hammer-cli-awesome'
      end

      # type of the translation files, 'mo' files are default
      # you can also use 'po' and 'yaml' files for testing purposes
      def type
        :mo
      end

    end

  end
end

# register the domain
HammerCLI::I18n.add_domain(HammerCLIAwesomePlugin::I18n::LocaleDomain.new)
```

Then you have to export strings, translate them and place the files in a directory structure.
Inspiration on how to export the translations can be found in hammer's [Rakefile](../Rakefile).

Typical directory structure for translation files look like this:
```
locale
├── de
│   ├── hammer-cli-awesome.po
│   └── LC_MESSAGES
│       └── hammer-cli-awesome.mo
├── en
│   ├── hammer-cli-awesome.po
│   └── LC_MESSAGES
│       └── hammer-cli-awesome.mo
└── hammer-cli-awesome.pot
```


### Translation tips

When writing code with translations make sure you keep two following rules:

1) Don't use variables directly in the strings and make formatting substitutions outside the gettext function `_("...")`.
```ruby
# WRONG
puts _("Hello #{name}")
puts _("Hello %s" % name)
# CORRECT
puts _("Hello %s") % name
```

2) Use named placeholders when there is more than one replacement. Languages differ in their word order.
```ruby
# WRONG
puts _("Hello %s, it is %s" % [name, day])
# CORRECT
puts _("Hello %{name}, it is %{day}") % {:name => name, :day => day}
```


3) Don't use newlines in the gettext function. Strings with newlines are not extracted correctly.
```ruby
# WRONG
puts _("TODO: \n - make dishes\n - do shopping")
# CORRECT
puts _("TODO:") +
     "\n - " + _("make dishes") +
     "\n - " + _("do shopping")
```


4) Try setting `:mark_translated: true` to identify gaps in your translations.
This will wrap all translated strings with angle brackets '>message<'.
