#!/usr/bin/env bash
set -euo pipefail


#############
# VARIABLES #
#############

basedir=$(dirname "$0")
do_not_generate="ceph-common$|ceph-docker-common$" # pipe separated list of roles we don't want to generate sample file, MUST end with '$', e.g: 'foo$|bar$'


#############
# FUNCTIONS #
#############

populate_header () {
  for i in $output; do
    cat <<EOF > "$basedir"/group_vars/"$i"
---
# Variables here are applicable to all host groups NOT roles

# This sample file generated by $(basename "$0")

# Dummy variable to avoid error because ansible does not recognize the
# file as a good configuration file when no variable in it.
dummy:

EOF
  done
}

generate_group_vars_file () {
  for i in $output; do
    if [ "$(uname)" == "Darwin" ]; then
      sed '/^---/d; s/^\([A-Za-z[:space:]]\)/#\1/' \
        "$defaults" >> "$basedir"/group_vars/"$i"
      echo >> "$basedir"/group_vars/"$i"
    elif [ "$(uname -s)" == "Linux" ]; then
      sed '/^---/d; s/^\([A-Za-z[:space:]].\+\)/#\1/' \
        "$defaults" >> "$basedir"/group_vars/"$i"
      echo >> "$basedir"/group_vars/"$i"
    else
      echo "Unsupported platform"
      exit 1
    fi
  done
}

rhcs_edits () {
  tail -n +1 rhcs_edits.txt | while IFS= read -r option; do
    sed -i "s|#${option% *} .*|${option}|" group_vars/rhcs.yml.sample
  done
}

########
# MAIN #
########

for role in "$basedir"/roles/ceph-*; do
  rolename=$(basename "$role")

  if [[ $rolename == "ceph-defaults" ]]; then
    output="all.yml.sample rhcs.yml.sample"
  elif [[ $rolename == "ceph-agent" ]]; then
    output="agent.yml.sample"
  elif [[ $rolename == "ceph-fetch-keys" ]]; then
    output="ceph-fetch-keys.yml.sample"
  elif [[ $rolename == "ceph-rbd-mirror" ]]; then
    output="rbdmirrors.yml.sample"
  elif [[ $rolename == "ceph-iscsi-gw" ]]; then
    output="iscsigws.yml.sample"
  else
    output="${rolename:5}s.yml.sample"
  fi

  defaults="$role"/defaults/main.yml
  if [[ ! -f $defaults ]]; then
    continue
  fi

  if ! echo "$rolename" | grep -qE "$do_not_generate"; then
    populate_header
    generate_group_vars_file
  fi
done

rhcs_edits
