/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.cxf.rs.security.jose.jws;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;

public class JwsInputStream extends FilterInputStream {
    private JwsVerificationSignature signature;
    private byte[] signatureBytes;
    private boolean verifyOnLastRead;
    public JwsInputStream(InputStream out, 
                          JwsVerificationSignature signature,
                          byte[] signatureBytes,
                          boolean verifyOnLastRead) {
        super(out);
        this.signature = signature;
        this.signatureBytes = signatureBytes;
        this.verifyOnLastRead = verifyOnLastRead;
    }

    @Override
    public int read() throws IOException {
        int value = super.read();
        if (value != -1) {
            byte[] bytes = ByteBuffer.allocate(Integer.SIZE / 8).putInt(value).array();
            signature.update(bytes, 0, bytes.length);
        } else {
            verify();
        }
        return value;
    }
 
    public int read(byte b[], int off, int len) throws IOException {
        int num = in.read(b, off, len);
        if (num != -1) {
            signature.update(b, off, num);
        } else {
            verify();
        }
        return num;
    }
    
    private void verify() {
        if (verifyOnLastRead && !signature.verify(signatureBytes)) {
            throw new JwsException(JwsException.Error.INVALID_SIGNATURE);    
        }
    }
}
