// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/ChargedFinalState.hh"

namespace Rivet {


  /// @brief Charged particle spectra at 5.2, 6.5 and 29 GeV
  class MARKII_1982_I178416 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(MARKII_1982_I178416);


    /// @name Analysis methods
    ///@{

    /// Book histograms and initialise projections before the run
    void init() {
      const ChargedFinalState fs;
      declare(fs, "FS");
      size_t ih = 0;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_h[en], 1, 1, ++ih);
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const ChargedFinalState& fs = apply<ChargedFinalState>(event, "FS");
      if (fs.particles().size()==2 && fs.particles()[0].abspid()==13 && fs.particles()[1].abspid()==13) vetoEvent;
      for (const Particle& p : fs.particles()) {
        const Vector3 mom3 = p.p3();
        double pp = mom3.mod();
        double x = 2.*pp/sqrtS();
        _h[_sqs]->fill(x);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for (auto& item : _h) {
        const double en = stod(item.first)*MeV;
        scale(item.second, crossSection()*sqr(en)/sumOfWeights()/microbarn);
      }
    }

    ///@}


    /// @name Histograms
    ///@{
    map<string,Histo1DPtr> _h;
    string _sqs = "";
    ///@}


  };


  RIVET_DECLARE_PLUGIN(MARKII_1982_I178416);

}
