/*
 * Copyright (c) 2023 One Identity LLC.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 */

#include <syslog-ng-config.h>

#if SYSLOG_NG_HAVE_ZLIB

#include <criterion/criterion.h>
#include <criterion/redirect.h>
#include "compression.h"

#if SYSLOG_NG_HTTP_COMPRESSION_ENABLED

char *test_message =
  "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.";

guint8 test_message_deflated_bytes[] =
{
  0x78, 0x9c, 0x35, 0x90, 0xc1, 0x71, 0x43, 0x31, 0x08, 0x44, 0x5b, 0xd9, 0x02,
  0x3c, 0xbf, 0x8a, 0xe4, 0x96, 0x6b, 0x0a, 0x20, 0x88, 0xef, 0x30, 0x23, 0x09,
  0x59, 0x02, 0x8f, 0xcb, 0x0f, 0xca, 0x4f, 0x6e, 0x42, 0xc0, 0xb2, 0xfb, 0x3e,
  0x6c, 0x4a, 0x83, 0x8e, 0x15, 0x0d, 0xc5, 0xaa, 0x4d, 0x2c, 0x75, 0x50, 0x13,
  0xbf, 0x81, 0xad, 0x2f, 0x61, 0x17, 0x8f, 0x09, 0x2a, 0x3a, 0x74, 0xb1, 0xf6,
  0x3b, 0xa4, 0x6a, 0x36, 0x97, 0x94, 0x5c, 0x80, 0x68, 0xac, 0x66, 0x05, 0x2e,
  0x6d, 0xe4, 0xb2, 0x76, 0xd6, 0xa2, 0x25, 0xba, 0x23, 0x1c, 0x95, 0xbe, 0x52,
  0x1e, 0xe2, 0x97, 0xb4, 0xa0, 0xd1, 0xbd, 0x13, 0xa8, 0xea, 0x23, 0xe8, 0xc0,
  0xa7, 0x43, 0xba, 0xb6, 0xd4, 0x46, 0xd3, 0xfd, 0x78, 0x66, 0x49, 0xed, 0x86,
  0x47, 0xe8, 0x42, 0xb7, 0xe5, 0x33, 0x0a, 0xe4, 0x25, 0x93, 0xd5, 0xc9, 0xd5,
  0x3a, 0xa2, 0x56, 0x6a, 0x6c, 0x97, 0xf2, 0x1e, 0xd2, 0xa5, 0xfb, 0xd2, 0xaf,
  0xa4, 0x8e, 0x1c, 0x86, 0x50, 0x1a, 0x6f, 0xe9, 0xc9, 0xae, 0x00, 0x79, 0xca,
  0x0f, 0xbc, 0x6d, 0x49, 0x0a, 0x17, 0xe8, 0x8c, 0x74, 0x72, 0x65, 0xd5, 0x8e,
  0x29, 0x63, 0xca, 0xb7, 0xf4, 0x22, 0x33, 0x83, 0xe7, 0xc7, 0xd3, 0x6a, 0x8c,
  0x3c, 0x27, 0x69, 0x27, 0x93, 0x42, 0xd6, 0x12, 0xb0, 0xd6, 0xfa, 0x4f, 0x28,
  0x03, 0x05, 0xce, 0xb8, 0x2b, 0x39, 0xfa, 0x36, 0x84, 0x41, 0x33, 0x8b, 0x98,
  0x07, 0xde, 0x5f, 0x2c, 0xc3, 0x25, 0x36, 0xc6, 0x64, 0x60, 0xcc, 0x24, 0x9c,
  0x73, 0x1c, 0x43, 0x0b, 0xf9, 0xde, 0xc8, 0x14, 0x63, 0x9a, 0x16, 0xe9, 0x9b,
  0xe2, 0x26, 0x95, 0x47, 0x39, 0xea, 0xa0, 0x9d, 0x1b, 0x76, 0x9e, 0xca, 0x4a,
  0x28, 0xb2, 0x64, 0xee, 0x6e, 0xb3, 0xba, 0x6d, 0xd0, 0x06, 0xa4, 0x89, 0x63,
  0xfd, 0x71, 0x8d, 0x76, 0xfc, 0x00, 0xa0, 0x5c, 0xa5, 0x09
};

guint test_message_deflated_length = G_N_ELEMENTS(test_message_deflated_bytes);

typedef enum _GzipOSID
{
  GZIP_OS_UNIX = 0x03,
  GZIP_OS_MACOS = 0x13,
  GZIP_OS_UNKNOWN = 0xff
} GzipOSID;

guint8 test_message_gzipped_bytes[] =
{
  0x1f, 0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x35, 0x90, 0xc1,
  0x71, 0x43, 0x31, 0x08, 0x44, 0x5b, 0xd9, 0x02, 0x3c, 0xbf, 0x8a, 0xe4, 0x96,
  0x6b, 0x0a, 0x20, 0x88, 0xef, 0x30, 0x23, 0x09, 0x59, 0x02, 0x8f, 0xcb, 0x0f,
  0xca, 0x4f, 0x6e, 0x42, 0xc0, 0xb2, 0xfb, 0x3e, 0x6c, 0x4a, 0x83, 0x8e, 0x15,
  0x0d, 0xc5, 0xaa, 0x4d, 0x2c, 0x75, 0x50, 0x13, 0xbf, 0x81, 0xad, 0x2f, 0x61,
  0x17, 0x8f, 0x09, 0x2a, 0x3a, 0x74, 0xb1, 0xf6, 0x3b, 0xa4, 0x6a, 0x36, 0x97,
  0x94, 0x5c, 0x80, 0x68, 0xac, 0x66, 0x05, 0x2e, 0x6d, 0xe4, 0xb2, 0x76, 0xd6,
  0xa2, 0x25, 0xba, 0x23, 0x1c, 0x95, 0xbe, 0x52, 0x1e, 0xe2, 0x97, 0xb4, 0xa0,
  0xd1, 0xbd, 0x13, 0xa8, 0xea, 0x23, 0xe8, 0xc0, 0xa7, 0x43, 0xba, 0xb6, 0xd4,
  0x46, 0xd3, 0xfd, 0x78, 0x66, 0x49, 0xed, 0x86, 0x47, 0xe8, 0x42, 0xb7, 0xe5,
  0x33, 0x0a, 0xe4, 0x25, 0x93, 0xd5, 0xc9, 0xd5, 0x3a, 0xa2, 0x56, 0x6a, 0x6c,
  0x97, 0xf2, 0x1e, 0xd2, 0xa5, 0xfb, 0xd2, 0xaf, 0xa4, 0x8e, 0x1c, 0x86, 0x50,
  0x1a, 0x6f, 0xe9, 0xc9, 0xae, 0x00, 0x79, 0xca, 0x0f, 0xbc, 0x6d, 0x49, 0x0a,
  0x17, 0xe8, 0x8c, 0x74, 0x72, 0x65, 0xd5, 0x8e, 0x29, 0x63, 0xca, 0xb7, 0xf4,
  0x22, 0x33, 0x83, 0xe7, 0xc7, 0xd3, 0x6a, 0x8c, 0x3c, 0x27, 0x69, 0x27, 0x93,
  0x42, 0xd6, 0x12, 0xb0, 0xd6, 0xfa, 0x4f, 0x28, 0x03, 0x05, 0xce, 0xb8, 0x2b,
  0x39, 0xfa, 0x36, 0x84, 0x41, 0x33, 0x8b, 0x98, 0x07, 0xde, 0x5f, 0x2c, 0xc3,
  0x25, 0x36, 0xc6, 0x64, 0x60, 0xcc, 0x24, 0x9c, 0x73, 0x1c, 0x43, 0x0b, 0xf9,
  0xde, 0xc8, 0x14, 0x63, 0x9a, 0x16, 0xe9, 0x9b, 0xe2, 0x26, 0x95, 0x47, 0x39,
  0xea, 0xa0, 0x9d, 0x1b, 0x76, 0x9e, 0xca, 0x4a, 0x28, 0xb2, 0x64, 0xee, 0x6e,
  0xb3, 0xba, 0x6d, 0xd0, 0x06, 0xa4, 0x89, 0x63, 0xfd, 0x71, 0x8d, 0x76, 0xfc,
  0x00, 0xbd, 0xc5, 0xb2, 0x98, 0xbd, 0x01, 0x00, 0x00
};

guint test_message_gzipped_length = G_N_ELEMENTS(test_message_gzipped_bytes);

void replace_gzip_header_os_id(guint8 *gzip_message)
{
  guint8 osid = GZIP_OS_UNKNOWN;

#if defined(__unix__) && defined(__linux__)
  osid = GZIP_OS_UNIX;
#endif
#ifdef __APPLE__
  osid = GZIP_OS_MACOS;
#endif

  // NOTE: gzip header OS ID is byte 9
  gzip_message[9] = osid;
}

// Needed because Criterion cr_assert_str_eq does not like random non-terminating \0s in a string.
static inline void
test_compression_results(GString *compression_result, const guint8 *expected_byte_array, const uint expected_size)
{
  cr_assert_eq(compression_result->len, expected_size);
  for(int i = 0; i < compression_result->len; i++)
    {
      cr_assert_eq((guint8) compression_result->str[i], expected_byte_array[i]);
    }
}

Compressor *compressor;
GString *input, *result;

static void
setup(void)
{
  cr_redirect_stdout();
  cr_redirect_stderr();
  input = g_string_new(test_message);
}

static void
teardown(void)
{
  g_string_free(input, 0);
}

TestSuite(compression, .init = setup, .fini = teardown);

Test(compression, compressor_gzip_compression)
{
  replace_gzip_header_os_id(test_message_gzipped_bytes);
  compressor = gzip_compressor_new();
  cr_assert_not_null(compressor);
  result = g_string_new("");
  cr_assert(compressor_compress(compressor, result, input));
  test_compression_results(result, test_message_gzipped_bytes, test_message_gzipped_length);
  compressor_free(compressor);
  g_string_free(result, TRUE);
}

Test(compression, compressor_deflate_compression)
{
  compressor = deflate_compressor_new();
  cr_assert_not_null(compressor);
  result = g_string_new("");
  cr_assert(compressor_compress(compressor, result, input));
  test_compression_results(result, test_message_deflated_bytes, test_message_deflated_length);
  compressor_free(compressor);
  g_string_free(result, TRUE);
}
#endif

#endif
