"""
This module benchmarks :class:`.PathSpec` using ~15 patterns against ~400 files.
"""

from functools import (
	partial)

import pytest
from pytest_benchmark.fixture import (
	BenchmarkFixture)

from pathspec import (
	PathSpec)
from pathspec._backends.simple.pathspec import (
	SimplePsBackend)

from benchmarks.hyperscan_pathspec_r1 import (
	HyperscanPsR1BlockClosureBackend,
	HyperscanPsR1BlockStateBackend,
	HyperscanPsR1StreamClosureBackend)

GROUP = "PathSpec.match_files(): 15 lines, 400 files"


# Hyperscan backend.

@pytest.mark.benchmark(group=GROUP)
def bench_hs_r1_block_closure(
	benchmark: BenchmarkFixture,
	flit_files: set[str],
	flit_gi_lines_all: list[str],
):
	spec = PathSpec.from_lines(
		'gitwildmatch',
		flit_gi_lines_all,
		backend='hyperscan',
		_test_backend_factory=HyperscanPsR1BlockClosureBackend,
	)
	benchmark(run_match, spec, flit_files)


@pytest.mark.benchmark(group=GROUP)
def bench_hs_r1_block_state(
	benchmark: BenchmarkFixture,
	flit_files: set[str],
	flit_gi_lines_all: list[str],
):
	spec = PathSpec.from_lines(
		'gitwildmatch',
		flit_gi_lines_all,
		backend='hyperscan',
		_test_backend_factory=HyperscanPsR1BlockStateBackend,
	)
	benchmark(run_match, spec, flit_files)


@pytest.mark.benchmark(group=GROUP)
def bench_hs_r1_stream_closure(
	benchmark: BenchmarkFixture,
	flit_files: set[str],
	flit_gi_lines_all: list[str],
):
	spec = PathSpec.from_lines(
		'gitwildmatch',
		flit_gi_lines_all,
		backend='hyperscan',
		_test_backend_factory=HyperscanPsR1StreamClosureBackend,
	)
	benchmark(run_match, spec, flit_files)


@pytest.mark.benchmark(group=GROUP)
def bench_hs_v1(
	benchmark: BenchmarkFixture,
	flit_files: set[str],
	flit_gi_lines_all: list[str],
):
	spec = PathSpec.from_lines(
		'gitwildmatch',
		flit_gi_lines_all,
		backend='hyperscan',
	)
	benchmark(run_match, spec, flit_files)


# Re2 backend.

@pytest.mark.benchmark(group=GROUP)
def bench_re2_v1(
	benchmark: BenchmarkFixture,
	flit_files: set[str],
	flit_gi_lines_all: list[str],
):
	spec = PathSpec.from_lines(
		'gitwildmatch',
		flit_gi_lines_all,
		backend='re2',
	)
	benchmark(run_match, spec, flit_files)


# Simple backend.

@pytest.mark.benchmark(group=GROUP)
def bench_sm_filtered(
	benchmark: BenchmarkFixture,
	flit_files: set[str],
	flit_gi_lines_all: list[str],
):
	spec = PathSpec.from_lines(
		'gitwildmatch',
		flit_gi_lines_all,
		backend='simple',
		_test_backend_factory=partial(SimplePsBackend, no_reverse=True)
	)
	benchmark(run_match, spec, flit_files)


@pytest.mark.benchmark(group=GROUP)
def bench_sm_filtered_reversed(
	benchmark: BenchmarkFixture,
	flit_files: set[str],
	flit_gi_lines_all: list[str],
):
	spec = PathSpec.from_lines(
		'gitwildmatch',
		flit_gi_lines_all,
		backend='simple',
		_test_backend_factory=SimplePsBackend,
	)
	benchmark(run_match, spec, flit_files)


@pytest.mark.benchmark(group=GROUP)
def bench_sm_unfiltered(
	benchmark: BenchmarkFixture,
	flit_files: set[str],
	flit_gi_lines_all: list[str],
):
	spec = PathSpec.from_lines(
		'gitwildmatch',
		flit_gi_lines_all,
		backend='simple',
		_test_backend_factory=partial(SimplePsBackend, no_filter=True, no_reverse=True)
	)
	benchmark(run_match, spec, flit_files)


@pytest.mark.benchmark(group=GROUP)
def bench_sm_unfiltered_reversed(
	benchmark: BenchmarkFixture,
	flit_files: set[str],
	flit_gi_lines_all: list[str],
):
	spec = PathSpec.from_lines(
		'gitwildmatch',
		flit_gi_lines_all,
		backend='simple',
		_test_backend_factory=partial(SimplePsBackend, no_filter=True)
	)
	benchmark(run_match, spec, flit_files)


@pytest.mark.benchmark(group=GROUP)
def bench_sm_v1(
	benchmark: BenchmarkFixture,
	flit_files: set[str],
	flit_gi_lines_all: list[str],
):
	spec = PathSpec.from_lines(
		'gitwildmatch',
		flit_gi_lines_all,
		backend='simple',
	)
	benchmark(run_match, spec, flit_files)


def run_match(spec: PathSpec, files: set[str]):
	for _ in spec.match_files(files):
		pass
