/*
This file is part of Telegram Desktop,
the official desktop application for the Telegram messaging service.

For license and copyright information please follow this link:
https://github.com/telegramdesktop/tdesktop/blob/master/LEGAL
*/
#pragma once

namespace RoundVideoData {

constexpr auto kLogoSize = 80;

struct LogoRLENode final {
	uint16_t count;
	uint8_t value;
};
using LogoRLEFrame = std::vector<LogoRLENode>;

void DecompressLogoRLEFrame(
		const std::vector<LogoRLENode> &rleFrame,
		uint8_t outFrame[kLogoSize][kLogoSize]) {
	auto pos = size_t(0);
	for (const auto &node : rleFrame) {
		for (auto i = uint16_t(0); i < node.count; ++i) {
			if (pos >= kLogoSize * kLogoSize) {
				break;
			}
			const auto y = int(pos / kLogoSize);
			const auto x = int(pos % kLogoSize);
			outFrame[y][x] = node.value;
			pos++;
		}
	}
}

const auto kLogoFrames = std::array<LogoRLEFrame, 27>{ {

	LogoRLEFrame{ { 997, 0 }, { 1, 9 }, { 1, 4 }, { 77, 0 }, { 1, 1 }, { 1, 27 }, { 78, 0 }, { 1, 32 }, { 1, 11 }, { 77, 0 }, { 1, 14 }, { 1, 44 }, { 77, 0 }, { 1, 2 }, { 1, 54 }, { 1, 17 }, { 77, 0 }, { 1, 33 }, { 1, 47 }, { 77, 0 }, { 1, 9 }, { 1, 61 }, { 1, 13 }, { 77, 0 }, { 1, 45 }, { 1, 40 }, { 77, 0 }, { 1, 19 }, { 1, 60 }, { 1, 6 }, { 76, 0 }, { 1, 2 }, { 1, 54 }, { 1, 31 }, { 77, 0 }, { 1, 27 }, { 1, 54 }, { 1, 2 }, { 76, 0 }, { 1, 1 }, { 1, 55 }, { 1, 15 }, { 9, 0 }, { 1, 53 }, { 1, 196 }, { 1, 65 }, { 65, 0 }, { 1, 24 }, { 1, 35 }, { 9, 0 }, { 1, 32 }, { 1, 238 }, { 1, 255 }, { 1, 115 }, { 65, 0 }, { 1, 41 }, { 1, 2 }, { 8, 0 }, { 1, 4 }, { 1, 199 }, { 2, 255 }, { 1, 121 }, { 20, 0 }, { 1, 2 }, { 1, 4 }, { 42, 0 }, { 1, 16 }, { 1, 9 }, { 9, 0 }, { 1, 135 }, { 3, 255 }, { 1, 98 }, { 19, 0 }, { 2, 19 }, { 43, 0 }, { 1, 7 }, { 9, 0 }, { 1, 68 }, { 1, 253 }, { 3, 255 }, { 1, 65 }, { 17, 0 }, { 1, 8 }, { 1, 43 }, { 1, 20 }, { 53, 0 }, { 1, 22 }, { 1, 231 }, { 3, 255 }, { 1, 217 }, { 1, 3 }, { 16, 0 }, { 1, 30 }, { 1, 56 }, { 1, 15 }, { 53, 0 }, { 1, 1 }, { 1, 183 }, { 3, 255 }, { 1, 250 }, { 1, 67 }, { 15, 0 }, { 1, 8 }, { 1, 48 }, { 1, 52 }, { 1, 10 }, { 54, 0 }, { 1, 115 }, { 4, 255 }, { 1, 91 }, { 15, 0 }, { 1, 23 }, { 1, 59 }, { 1, 39 }, { 1, 2 }, { 54, 0 }, { 1, 52 }, { 1, 250 }, { 3, 255 }, { 1, 114 }, { 14, 0 }, { 1, 4 }, { 1, 41 }, { 1, 60 }, { 1, 22 }, { 55, 0 }, { 1, 13 }, { 1, 219 }, { 3, 255 }, { 1, 139 }, { 14, 0 }, { 1, 10 }, { 1, 54 }, { 1, 51 }, { 1, 9 }, { 56, 0 }, { 1, 164 }, { 3, 255 }, { 1, 162 }, { 1, 1 }, { 1, 0 }, { 1, 102 }, { 1, 222 }, { 1, 236 }, { 1, 85 }, { 8, 0 }, { 1, 15 }, { 1, 54 }, { 1, 27 }, { 57, 0 }, { 1, 95 }, { 3, 255 }, { 1, 183 }, { 1, 5 }, { 1, 2 }, { 1, 152 }, { 3, 255 }, { 1, 164 }, { 7, 0 }, { 1, 22 }, { 1, 36 }, { 1, 4 }, { 57, 0 }, { 1, 38 }, { 1, 243 }, { 2, 255 }, { 1, 201 }, { 1, 11 }, { 1, 7 }, { 1, 172 }, { 4, 255 }, { 1, 188 }, { 6, 0 }, { 1, 16 }, { 1, 12 }, { 58, 0 }, { 1, 6 }, { 1, 206 }, { 2, 255 }, { 1, 217 }, { 1, 20 }, { 1, 14 }, { 1, 190 }, { 5, 255 }, { 1, 213 }, { 5, 0 }, { 1, 1 }, { 60, 0 }, { 1, 144 }, { 2, 255 }, { 1, 230 }, { 1, 32 }, { 1, 23 }, { 1, 207 }, { 6, 255 }, { 1, 238 }, { 65, 0 }, { 1, 76 }, { 2, 255 }, { 1, 240 }, { 1, 46 }, { 1, 35 }, { 1, 220 }, { 8, 255 }, { 1, 8 }, { 63, 0 }, { 1, 26 }, { 1, 235 }, { 1, 255 }, { 1, 248 }, { 1, 64 }, { 1, 48 }, { 1, 232 }, { 9, 255 }, { 1, 31 }, { 62, 0 }, { 1, 2 }, { 1, 191 }, { 1, 255 }, { 1, 253 }, { 1, 83 }, { 1, 64 }, { 1, 241 }, { 10, 255 }, { 1, 56 }, { 62, 0 }, { 1, 124 }, { 2, 255 }, { 1, 106 }, { 1, 83 }, { 1, 248 }, { 11, 255 }, { 1, 81 }, { 61, 0 }, { 1, 59 }, { 1, 252 }, { 1, 255 }, { 1, 130 }, { 1, 103 }, { 1, 253 }, { 12, 255 }, { 1, 105 }, { 60, 0 }, { 1, 17 }, { 1, 225 }, { 1, 255 }, { 1, 226 }, { 1, 143 }, { 14, 255 }, { 1, 103 }, { 60, 0 }, { 1, 173 }, { 17, 255 }, { 1, 223 }, { 1, 12 }, { 59, 0 }, { 1, 105 }, { 17, 255 }, { 1, 206 }, { 1, 30 }, { 59, 0 }, { 1, 44 }, { 1, 247 }, { 15, 255 }, { 1, 204 }, { 1, 95 }, { 1, 4 }, { 59, 0 }, { 1, 9 }, { 1, 213 }, { 12, 255 }, { 1, 252 }, { 1, 191 }, { 1, 110 }, { 1, 30 }, { 62, 0 }, { 1, 154 }, { 10, 255 }, { 1, 247 }, { 1, 178 }, { 1, 97 }, { 1, 20 }, { 64, 0 }, { 1, 85 }, { 8, 255 }, { 1, 240 }, { 1, 165 }, { 1, 84 }, { 1, 12 }, { 66, 0 }, { 1, 32 }, { 1, 240 }, { 5, 255 }, { 1, 232 }, { 1, 153 }, { 1, 71 }, { 1, 6 }, { 68, 0 }, { 1, 1 }, { 1, 196 }, { 3, 255 }, { 1, 221 }, { 1, 140 }, { 1, 59 }, { 1, 2 }, { 71, 0 }, { 1, 49 }, { 1, 253 }, { 1, 207 }, { 1, 127 }, { 1, 46 }, { 76, 0 }, { 1, 6 }, { 508, 0 }, { 1, 6 }, { 1, 1 }, { 76, 0 }, { 1, 7 }, { 1, 29 }, { 1, 9 }, { 75, 0 }, { 1, 6 }, { 1, 36 }, { 1, 41 }, { 1, 3 }, { 74, 0 }, { 1, 1 }, { 1, 31 }, { 1, 59 }, { 1, 29 }, { 75, 0 }, { 1, 17 }, { 1, 53 }, { 1, 51 }, { 1, 13 }, { 74, 0 }, { 1, 6 }, { 1, 42 }, { 1, 60 }, { 1, 29 }, { 75, 0 }, { 1, 25 }, { 1, 60 }, { 1, 47 }, { 1, 9 }, { 74, 0 }, { 1, 3 }, { 1, 41 }, { 1, 52 }, { 1, 21 }, { 75, 0 }, { 1, 12 }, { 1, 43 }, { 1, 21 }, { 76, 0 }, { 1, 15 }, { 1, 18 }, { 77, 0 }, { 1, 1 }, { 843, 0 } }, 
	LogoRLEFrame{ { 600, 0 }, { 1, 1 }, { 79, 0 }, { 1, 7 }, { 78, 0 }, { 1, 14 }, { 1, 1 }, { 77, 0 }, { 1, 7 }, { 1, 16 }, { 77, 0 }, { 1, 1 }, { 1, 26 }, { 1, 3 }, { 77, 0 }, { 1, 18 }, { 1, 19 }, { 77, 0 }, { 1, 6 }, { 1, 29 }, { 1, 3 }, { 77, 0 }, { 1, 24 }, { 1, 16 }, { 77, 0 }, { 1, 12 }, { 1, 28 }, { 1, 1 }, { 76, 0 }, { 1, 2 }, { 1, 28 }, { 1, 12 }, { 77, 0 }, { 1, 17 }, { 1, 25 }, { 77, 0 }, { 1, 3 }, { 1, 29 }, { 1, 7 }, { 77, 0 }, { 1, 16 }, { 1, 17 }, { 77, 0 }, { 1, 1 }, { 1, 24 }, { 1, 1 }, { 77, 0 }, { 1, 13 }, { 1, 5 }, { 77, 0 }, { 1, 1 }, { 1, 8 }, { 78, 0 }, { 1, 1 }, { 90, 0 }, { 1, 14 }, { 1, 167 }, { 1, 127 }, { 76, 0 }, { 1, 2 }, { 1, 189 }, { 1, 255 }, { 1, 196 }, { 76, 0 }, { 1, 124 }, { 2, 255 }, { 1, 204 }, { 75, 0 }, { 1, 58 }, { 1, 252 }, { 2, 255 }, { 1, 185 }, { 74, 0 }, { 1, 16 }, { 1, 224 }, { 3, 255 }, { 1, 155 }, { 74, 0 }, { 1, 171 }, { 4, 255 }, { 1, 64 }, { 73, 0 }, { 1, 101 }, { 4, 255 }, { 1, 167 }, { 73, 0 }, { 1, 42 }, { 1, 245 }, { 3, 255 }, { 1, 192 }, { 1, 7 }, { 72, 0 }, { 1, 8 }, { 1, 210 }, { 3, 255 }, { 1, 209 }, { 1, 15 }, { 57, 0 }, { 1, 5 }, { 15, 0 }, { 1, 148 }, { 3, 255 }, { 1, 223 }, { 1, 25 }, { 57, 0 }, { 1, 14 }, { 1, 1 }, { 14, 0 }, { 1, 80 }, { 3, 255 }, { 1, 235 }, { 1, 38 }, { 1, 0 }, { 1, 26 }, { 1, 159 }, { 1, 214 }, { 1, 142 }, { 52, 0 }, { 1, 4 }, { 1, 20 }, { 14, 0 }, { 1, 28 }, { 1, 237 }, { 2, 255 }, { 1, 245 }, { 1, 54 }, { 1, 0 }, { 1, 48 }, { 1, 231 }, { 3, 255 }, { 1, 7 }, { 51, 0 }, { 1, 29 }, { 1, 5 }, { 13, 0 }, { 1, 3 }, { 1, 193 }, { 2, 255 }, { 1, 251 }, { 1, 74 }, { 1, 0 }, { 1, 64 }, { 1, 241 }, { 4, 255 }, { 1, 32 }, { 50, 0 }, { 1, 14 }, { 1, 30 }, { 14, 0 }, { 1, 126 }, { 3, 255 }, { 1, 95 }, { 1, 0 }, { 1, 83 }, { 1, 248 }, { 5, 255 }, { 1, 59 }, { 49, 0 }, { 1, 1 }, { 1, 37 }, { 1, 12 }, { 13, 0 }, { 1, 60 }, { 1, 252 }, { 2, 255 }, { 1, 119 }, { 1, 0 }, { 1, 105 }, { 1, 253 }, { 6, 255 }, { 1, 85 }, { 49, 0 }, { 1, 21 }, { 1, 32 }, { 13, 0 }, { 1, 17 }, { 1, 225 }, { 2, 255 }, { 1, 144 }, { 1, 0 }, { 1, 128 }, { 8, 255 }, { 1, 111 }, { 48, 0 }, { 1, 3 }, { 1, 40 }, { 1, 11 }, { 13, 0 }, { 1, 173 }, { 2, 255 }, { 1, 167 }, { 1, 4 }, { 1, 152 }, { 9, 255 }, { 1, 138 }, { 48, 0 }, { 1, 25 }, { 1, 31 }, { 13, 0 }, { 1, 104 }, { 2, 255 }, { 1, 188 }, { 1, 14 }, { 1, 174 }, { 10, 255 }, { 1, 164 }, { 47, 0 }, { 1, 6 }, { 1, 41 }, { 1, 9 }, { 12, 0 }, { 1, 43 }, { 1, 246 }, { 1, 255 }, { 1, 206 }, { 1, 29 }, { 1, 192 }, { 11, 255 }, { 1, 191 }, { 47, 0 }, { 1, 28 }, { 1, 29 }, { 12, 0 }, { 1, 9 }, { 1, 211 }, { 1, 255 }, { 1, 221 }, { 1, 49 }, { 1, 209 }, { 12, 255 }, { 1, 218 }, { 46, 0 }, { 1, 5 }, { 1, 40 }, { 1, 5 }, { 12, 0 }, { 1, 151 }, { 2, 255 }, { 1, 125 }, { 1, 223 }, { 13, 255 }, { 1, 223 }, { 46, 0 }, { 1, 22 }, { 1, 20 }, { 12, 0 }, { 1, 82 }, { 18, 255 }, { 1, 121 }, { 45, 0 }, { 1, 1 }, { 1, 32 }, { 12, 0 }, { 1, 30 }, { 1, 238 }, { 17, 255 }, { 1, 139 }, { 1, 2 }, { 45, 0 }, { 1, 15 }, { 1, 8 }, { 11, 0 }, { 1, 3 }, { 1, 195 }, { 15, 255 }, { 1, 253 }, { 1, 186 }, { 1, 63 }, { 47, 0 }, { 1, 11 }, { 12, 0 }, { 1, 128 }, { 13, 255 }, { 1, 247 }, { 1, 179 }, { 1, 99 }, { 1, 22 }, { 48, 0 }, { 1, 1 }, { 12, 0 }, { 1, 62 }, { 1, 252 }, { 10, 255 }, { 1, 238 }, { 1, 163 }, { 1, 83 }, { 1, 12 }, { 63, 0 }, { 1, 18 }, { 1, 227 }, { 8, 255 }, { 1, 226 }, { 1, 146 }, { 1, 66 }, { 1, 4 }, { 66, 0 }, { 1, 175 }, { 6, 255 }, { 1, 209 }, { 1, 130 }, { 1, 50 }, { 69, 0 }, { 1, 100 }, { 3, 255 }, { 1, 252 }, { 1, 193 }, { 1, 113 }, { 1, 34 }, { 72, 0 }, { 1, 216 }, { 1, 246 }, { 1, 176 }, { 1, 97 }, { 1, 21 }, { 75, 0 }, { 1, 17 }, { 1, 4 }, { 349, 0 }, { 1, 18 }, { 1, 13 }, { 76, 0 }, { 1, 19 }, { 1, 42 }, { 1, 11 }, { 75, 0 }, { 1, 15 }, { 1, 49 }, { 1, 43 }, { 1, 3 }, { 74, 0 }, { 1, 5 }, { 1, 40 }, { 1, 59 }, { 1, 27 }, { 75, 0 }, { 1, 26 }, { 1, 59 }, { 1, 45 }, { 1, 7 }, { 74, 0 }, { 1, 12 }, { 1, 50 }, { 1, 57 }, { 1, 22 }, { 75, 0 }, { 1, 27 }, { 1, 60 }, { 1, 38 }, { 1, 4 }, { 74, 0 }, { 1, 4 }, { 1, 42 }, { 1, 38 }, { 1, 8 }, { 75, 0 }, { 1, 12 }, { 1, 29 }, { 1, 7 }, { 76, 0 }, { 1, 2 }, { 1, 5 }, { 999, 0 } }, 
	LogoRLEFrame{ { 2042, 0 }, { 1, 2 }, { 1, 144 }, { 1, 220 }, { 1, 25 }, { 1, 4 }, { 75, 0 }, { 1, 132 }, { 2, 255 }, { 1, 64 }, { 75, 0 }, { 1, 62 }, { 1, 253 }, { 2, 255 }, { 1, 55 }, { 74, 0 }, { 1, 16 }, { 1, 225 }, { 3, 255 }, { 1, 43 }, { 74, 0 }, { 1, 169 }, { 4, 255 }, { 1, 19 }, { 55, 0 }, { 2, 5 }, { 16, 0 }, { 1, 96 }, { 4, 255 }, { 1, 201 }, { 56, 0 }, { 1, 26 }, { 16, 0 }, { 1, 35 }, { 1, 243 }, { 3, 255 }, { 1, 252 }, { 1, 66 }, { 55, 0 }, { 1, 20 }, { 1, 21 }, { 15, 0 }, { 1, 4 }, { 1, 201 }, { 4, 255 }, { 1, 105 }, { 55, 0 }, { 1, 3 }, { 1, 50 }, { 1, 1 }, { 15, 0 }, { 1, 132 }, { 4, 255 }, { 1, 134 }, { 56, 0 }, { 1, 37 }, { 1, 32 }, { 15, 0 }, { 1, 62 }, { 1, 253 }, { 3, 255 }, { 1, 160 }, { 56, 0 }, { 1, 12 }, { 1, 60 }, { 1, 6 }, { 14, 0 }, { 1, 16 }, { 1, 225 }, { 3, 255 }, { 1, 183 }, { 1, 4 }, { 1, 0 }, { 1, 28 }, { 1, 132 }, { 1, 143 }, { 1, 31 }, { 51, 0 }, { 1, 46 }, { 1, 34 }, { 15, 0 }, { 1, 169 }, { 3, 255 }, { 1, 202 }, { 1, 11 }, { 1, 0 }, { 1, 73 }, { 1, 243 }, { 2, 255 }, { 1, 165 }, { 50, 0 }, { 1, 17 }, { 1, 59 }, { 1, 5 }, { 14, 0 }, { 1, 95 }, { 3, 255 }, { 1, 219 }, { 1, 21 }, { 1, 0 }, { 1, 94 }, { 1, 251 }, { 3, 255 }, { 1, 197 }, { 50, 0 }, { 1, 51 }, { 1, 32 }, { 14, 0 }, { 1, 35 }, { 1, 242 }, { 2, 255 }, { 1, 232 }, { 1, 34 }, { 1, 0 }, { 1, 118 }, { 5, 255 }, { 1, 229 }, { 49, 0 }, { 1, 24 }, { 1, 58 }, { 1, 3 }, { 13, 0 }, { 1, 4 }, { 1, 200 }, { 2, 255 }, { 1, 243 }, { 1, 51 }, { 1, 1 }, { 1, 143 }, { 7, 255 }, { 1, 7 }, { 48, 0 }, { 1, 55 }, { 1, 29 }, { 14, 0 }, { 1, 132 }, { 2, 255 }, { 1, 250 }, { 1, 70 }, { 1, 6 }, { 1, 167 }, { 8, 255 }, { 1, 38 }, { 47, 0 }, { 1, 20 }, { 1, 52 }, { 1, 1 }, { 13, 0 }, { 1, 61 }, { 1, 253 }, { 2, 255 }, { 1, 93 }, { 1, 13 }, { 1, 187 }, { 9, 255 }, { 1, 70 }, { 47, 0 }, { 1, 47 }, { 1, 15 }, { 13, 0 }, { 1, 16 }, { 1, 225 }, { 2, 255 }, { 1, 118 }, { 1, 23 }, { 1, 205 }, { 10, 255 }, { 1, 102 }, { 46, 0 }, { 1, 11 }, { 1, 37 }, { 14, 0 }, { 1, 168 }, { 2, 255 }, { 1, 145 }, { 1, 36 }, { 1, 221 }, { 11, 255 }, { 1, 134 }, { 46, 0 }, { 1, 29 }, { 1, 2 }, { 13, 0 }, { 1, 95 }, { 2, 255 }, { 1, 170 }, { 1, 53 }, { 1, 233 }, { 12, 255 }, { 1, 166 }, { 45, 0 }, { 1, 3 }, { 1, 10 }, { 13, 0 }, { 1, 35 }, { 1, 242 }, { 1, 255 }, { 1, 226 }, { 1, 89 }, { 1, 243 }, { 13, 255 }, { 1, 187 }, { 59, 0 }, { 1, 4 }, { 1, 200 }, { 18, 255 }, { 1, 136 }, { 59, 0 }, { 1, 131 }, { 18, 255 }, { 1, 190 }, { 1, 9 }, { 58, 0 }, { 1, 61 }, { 1, 253 }, { 16, 255 }, { 1, 237 }, { 1, 125 }, { 1, 4 }, { 58, 0 }, { 1, 16 }, { 1, 225 }, { 14, 255 }, { 1, 235 }, { 1, 161 }, { 1, 84 }, { 1, 11 }, { 60, 0 }, { 1, 168 }, { 12, 255 }, { 1, 213 }, { 1, 136 }, { 1, 59 }, { 1, 2 }, { 62, 0 }, { 1, 94 }, { 9, 255 }, { 1, 250 }, { 1, 188 }, { 1, 111 }, { 1, 34 }, { 65, 0 }, { 1, 34 }, { 1, 242 }, { 6, 255 }, { 1, 236 }, { 1, 162 }, { 1, 85 }, { 1, 14 }, { 67, 0 }, { 1, 1 }, { 1, 198 }, { 4, 255 }, { 1, 214 }, { 1, 137 }, { 1, 60 }, { 1, 3 }, { 70, 0 }, { 1, 67 }, { 1, 255 }, { 1, 251 }, { 1, 189 }, { 1, 112 }, { 1, 35 }, { 74, 0 }, { 1, 9 }, { 1, 57 }, { 1, 14 }, { 111, 0 }, { 1, 6 }, { 1, 11 }, { 76, 0 }, { 1, 5 }, { 1, 32 }, { 1, 20 }, { 75, 0 }, { 1, 4 }, { 1, 33 }, { 1, 51 }, { 1, 11 }, { 75, 0 }, { 1, 24 }, { 1, 57 }, { 1, 42 }, { 1, 3 }, { 74, 0 }, { 1, 11 }, { 1, 48 }, { 1, 56 }, { 1, 20 }, { 74, 0 }, { 1, 3 }, { 1, 35 }, { 1, 61 }, { 1, 38 }, { 1, 3 }, { 74, 0 }, { 1, 14 }, { 1, 55 }, { 1, 53 }, { 1, 15 }, { 75, 0 }, { 1, 28 }, { 1, 52 }, { 1, 24 }, { 75, 0 }, { 1, 4 }, { 1, 34 }, { 1, 23 }, { 76, 0 }, { 1, 4 }, { 1, 15 }, { 1156, 0 } }, 
	LogoRLEFrame{ { 1807, 0 }, { 2, 3 }, { 77, 0 }, { 1, 5 }, { 1, 34 }, { 74, 0 }, { 1, 15 }, { 1, 88 }, { 1, 9 }, { 1, 7 }, { 1, 52 }, { 1, 10 }, { 73, 0 }, { 1, 14 }, { 1, 210 }, { 1, 255 }, { 1, 94 }, { 1, 51 }, { 1, 33 }, { 57, 0 }, { 1, 2 }, { 16, 0 }, { 1, 160 }, { 2, 255 }, { 1, 151 }, { 1, 47 }, { 57, 0 }, { 1, 13 }, { 1, 1 }, { 15, 0 }, { 1, 80 }, { 3, 255 }, { 1, 178 }, { 1, 5 }, { 56, 0 }, { 1, 2 }, { 1, 27 }, { 15, 0 }, { 1, 21 }, { 1, 233 }, { 3, 255 }, { 1, 157 }, { 57, 0 }, { 1, 35 }, { 1, 9 }, { 15, 0 }, { 1, 175 }, { 4, 255 }, { 1, 143 }, { 56, 0 }, { 1, 13 }, { 1, 45 }, { 15, 0 }, { 1, 94 }, { 5, 255 }, { 1, 73 }, { 55, 0 }, { 1, 1 }, { 1, 52 }, { 1, 20 }, { 14, 0 }, { 1, 29 }, { 1, 240 }, { 4, 255 }, { 1, 199 }, { 1, 1 }, { 55, 0 }, { 1, 26 }, { 1, 52 }, { 14, 0 }, { 1, 1 }, { 1, 189 }, { 4, 255 }, { 1, 227 }, { 1, 24 }, { 55, 0 }, { 1, 3 }, { 1, 57 }, { 1, 20 }, { 14, 0 }, { 1, 109 }, { 4, 255 }, { 1, 241 }, { 1, 46 }, { 56, 0 }, { 1, 32 }, { 1, 50 }, { 14, 0 }, { 1, 39 }, { 1, 246 }, { 3, 255 }, { 1, 250 }, { 1, 68 }, { 56, 0 }, { 1, 6 }, { 1, 60 }, { 1, 18 }, { 13, 0 }, { 1, 3 }, { 1, 201 }, { 4, 255 }, { 1, 93 }, { 2, 0 }, { 1, 24 }, { 1, 97 }, { 1, 81 }, { 1, 1 }, { 51, 0 }, { 1, 38 }, { 1, 48 }, { 14, 0 }, { 1, 124 }, { 4, 255 }, { 1, 122 }, { 2, 0 }, { 1, 83 }, { 1, 240 }, { 2, 255 }, { 1, 113 }, { 50, 0 }, { 1, 7 }, { 1, 61 }, { 1, 14 }, { 13, 0 }, { 1, 50 }, { 1, 250 }, { 3, 255 }, { 1, 153 }, { 2, 0 }, { 1, 109 }, { 1, 253 }, { 3, 255 }, { 1, 164 }, { 50, 0 }, { 1, 33 }, { 1, 37 }, { 13, 0 }, { 1, 7 }, { 1, 212 }, { 3, 255 }, { 1, 180 }, { 1, 3 }, { 1, 1 }, { 1, 136 }, { 5, 255 }, { 1, 206 }, { 49, 0 }, { 1, 2 }, { 1, 52 }, { 1, 4 }, { 13, 0 }, { 1, 139 }, { 3, 255 }, { 1, 203 }, { 1, 11 }, { 1, 5 }, { 1, 163 }, { 6, 255 }, { 1, 247 }, { 1, 2 }, { 48, 0 }, { 1, 24 }, { 1, 20 }, { 13, 0 }, { 1, 61 }, { 1, 253 }, { 2, 255 }, { 1, 221 }, { 1, 22 }, { 1, 12 }, { 1, 186 }, { 8, 255 }, { 1, 35 }, { 48, 0 }, { 1, 27 }, { 13, 0 }, { 1, 12 }, { 1, 222 }, { 2, 255 }, { 1, 236 }, { 1, 38 }, { 1, 23 }, { 1, 205 }, { 9, 255 }, { 1, 78 }, { 47, 0 }, { 1, 6 }, { 1, 2 }, { 13, 0 }, { 1, 154 }, { 2, 255 }, { 1, 247 }, { 1, 58 }, { 1, 37 }, { 1, 222 }, { 10, 255 }, { 1, 120 }, { 61, 0 }, { 1, 74 }, { 2, 255 }, { 1, 253 }, { 1, 82 }, { 1, 55 }, { 1, 235 }, { 11, 255 }, { 1, 162 }, { 60, 0 }, { 1, 19 }, { 1, 230 }, { 2, 255 }, { 1, 110 }, { 1, 75 }, { 1, 245 }, { 12, 255 }, { 1, 205 }, { 60, 0 }, { 1, 169 }, { 2, 255 }, { 1, 180 }, { 1, 103 }, { 1, 251 }, { 13, 255 }, { 1, 242 }, { 59, 0 }, { 1, 89 }, { 19, 255 }, { 1, 232 }, { 58, 0 }, { 1, 26 }, { 1, 238 }, { 19, 255 }, { 1, 87 }, { 58, 0 }, { 1, 183 }, { 18, 255 }, { 1, 233 }, { 1, 78 }, { 58, 0 }, { 1, 104 }, { 16, 255 }, { 1, 243 }, { 1, 176 }, { 1, 98 }, { 1, 8 }, { 58, 0 }, { 1, 35 }, { 1, 244 }, { 13, 255 }, { 1, 209 }, { 1, 136 }, { 1, 64 }, { 1, 6 }, { 60, 0 }, { 1, 2 }, { 1, 196 }, { 10, 255 }, { 1, 238 }, { 1, 169 }, { 1, 97 }, { 1, 26 }, { 64, 0 }, { 1, 119 }, { 7, 255 }, { 1, 253 }, { 1, 201 }, { 1, 129 }, { 1, 57 }, { 1, 3 }, { 66, 0 }, { 1, 39 }, { 1, 249 }, { 4, 255 }, { 1, 232 }, { 1, 161 }, { 1, 89 }, { 1, 20 }, { 70, 0 }, { 1, 139 }, { 1, 255 }, { 1, 251 }, { 1, 193 }, { 1, 121 }, { 1, 49 }, { 1, 1 }, { 31, 0 }, { 1, 7 }, { 41, 0 }, { 1, 23 }, { 1, 62 }, { 1, 14 }, { 33, 0 }, { 1, 18 }, { 1, 24 }, { 1, 1 }, { 75, 0 }, { 1, 16 }, { 1, 46 }, { 1, 22 }, { 75, 0 }, { 1, 10 }, { 1, 46 }, { 1, 53 }, { 1, 11 }, { 74, 0 }, { 1, 2 }, { 1, 33 }, { 1, 60 }, { 1, 35 }, { 1, 2 }, { 74, 0 }, { 1, 19 }, { 1, 55 }, { 1, 51 }, { 1, 13 }, { 74, 0 }, { 1, 5 }, { 1, 43 }, { 1, 61 }, { 1, 30 }, { 1, 1 }, { 74, 0 }, { 1, 15 }, { 1, 55 }, { 1, 40 }, { 1, 9 }, { 75, 0 }, { 1, 29 }, { 1, 38 }, { 1, 9 }, { 75, 0 }, { 1, 4 }, { 1, 22 }, { 1, 8 }, { 77, 0 }, { 1, 2 }, { 1234, 0 } }, 
	LogoRLEFrame{ { 1649, 0 }, { 1, 23 }, { 1, 2 }, { 77, 0 }, { 1, 33 }, { 1, 24 }, { 77, 0 }, { 1, 33 }, { 1, 48 }, { 56, 0 }, { 2, 2 }, { 15, 0 }, { 1, 17 }, { 3, 0 }, { 1, 24 }, { 1, 59 }, { 1, 9 }, { 56, 0 }, { 1, 18 }, { 15, 0 }, { 1, 152 }, { 1, 255 }, { 1, 98 }, { 1, 0 }, { 1, 15 }, { 1, 61 }, { 1, 19 }, { 56, 0 }, { 1, 12 }, { 1, 20 }, { 14, 0 }, { 1, 98 }, { 2, 255 }, { 1, 169 }, { 1, 5 }, { 1, 58 }, { 1, 32 }, { 57, 0 }, { 1, 45 }, { 1, 1 }, { 13, 0 }, { 1, 24 }, { 1, 239 }, { 2, 255 }, { 1, 216 }, { 1, 43 }, { 1, 39 }, { 57, 0 }, { 1, 29 }, { 1, 33 }, { 14, 0 }, { 1, 173 }, { 3, 255 }, { 1, 250 }, { 1, 36 }, { 57, 0 }, { 1, 8 }, { 1, 59 }, { 1, 8 }, { 13, 0 }, { 1, 82 }, { 5, 255 }, { 1, 2 }, { 57, 0 }, { 1, 41 }, { 1, 39 }, { 13, 0 }, { 1, 16 }, { 1, 230 }, { 5, 255 }, { 1, 7 }, { 56, 0 }, { 1, 12 }, { 1, 61 }, { 1, 7 }, { 13, 0 }, { 1, 156 }, { 5, 255 }, { 1, 198 }, { 57, 0 }, { 1, 47 }, { 1, 36 }, { 13, 0 }, { 1, 66 }, { 6, 255 }, { 1, 85 }, { 56, 0 }, { 1, 18 }, { 1, 60 }, { 1, 6 }, { 12, 0 }, { 1, 10 }, { 1, 221 }, { 5, 255 }, { 1, 152 }, { 57, 0 }, { 1, 52 }, { 1, 34 }, { 13, 0 }, { 1, 140 }, { 5, 255 }, { 1, 192 }, { 1, 6 }, { 56, 0 }, { 1, 19 }, { 1, 57 }, { 1, 4 }, { 12, 0 }, { 1, 52 }, { 1, 252 }, { 4, 255 }, { 1, 216 }, { 1, 17 }, { 57, 0 }, { 1, 46 }, { 1, 22 }, { 12, 0 }, { 1, 5 }, { 1, 209 }, { 4, 255 }, { 1, 235 }, { 1, 35 }, { 3, 0 }, { 1, 23 }, { 1, 7 }, { 52, 0 }, { 1, 10 }, { 1, 46 }, { 13, 0 }, { 1, 123 }, { 4, 255 }, { 1, 247 }, { 1, 58 }, { 2, 0 }, { 1, 48 }, { 1, 206 }, { 1, 255 }, { 1, 246 }, { 1, 77 }, { 51, 0 }, { 1, 33 }, { 1, 7 }, { 12, 0 }, { 1, 40 }, { 1, 248 }, { 4, 255 }, { 1, 87 }, { 2, 0 }, { 1, 73 }, { 1, 244 }, { 3, 255 }, { 1, 173 }, { 50, 0 }, { 1, 4 }, { 1, 18 }, { 12, 0 }, { 1, 1 }, { 1, 196 }, { 4, 255 }, { 1, 121 }, { 2, 0 }, { 1, 100 }, { 1, 251 }, { 4, 255 }, { 1, 233 }, { 50, 0 }, { 1, 5 }, { 13, 0 }, { 1, 107 }, { 4, 255 }, { 1, 157 }, { 2, 0 }, { 1, 131 }, { 7, 255 }, { 1, 36 }, { 62, 0 }, { 1, 29 }, { 1, 242 }, { 3, 255 }, { 1, 188 }, { 2, 5 }, { 1, 160 }, { 8, 255 }, { 1, 95 }, { 62, 0 }, { 1, 181 }, { 3, 255 }, { 1, 213 }, { 1, 15 }, { 1, 14 }, { 1, 187 }, { 9, 255 }, { 1, 154 }, { 61, 0 }, { 1, 90 }, { 3, 255 }, { 1, 232 }, { 1, 32 }, { 1, 27 }, { 1, 209 }, { 10, 255 }, { 1, 214 }, { 60, 0 }, { 1, 20 }, { 1, 235 }, { 2, 255 }, { 1, 246 }, { 1, 54 }, { 1, 44 }, { 1, 227 }, { 12, 255 }, { 1, 18 }, { 59, 0 }, { 1, 165 }, { 3, 255 }, { 1, 82 }, { 1, 65 }, { 1, 240 }, { 13, 255 }, { 1, 76 }, { 58, 0 }, { 1, 74 }, { 3, 255 }, { 1, 166 }, { 1, 94 }, { 1, 249 }, { 14, 255 }, { 1, 134 }, { 57, 0 }, { 1, 13 }, { 1, 226 }, { 20, 255 }, { 1, 165 }, { 57, 0 }, { 1, 148 }, { 21, 255 }, { 1, 89 }, { 56, 0 }, { 1, 60 }, { 21, 255 }, { 1, 137 }, { 56, 0 }, { 1, 7 }, { 1, 215 }, { 18, 255 }, { 1, 252 }, { 1, 185 }, { 1, 68 }, { 57, 0 }, { 1, 132 }, { 15, 255 }, { 1, 253 }, { 1, 205 }, { 1, 140 }, { 1, 75 }, { 1, 14 }, { 58, 0 }, { 1, 46 }, { 1, 250 }, { 12, 255 }, { 1, 209 }, { 1, 144 }, { 1, 79 }, { 1, 17 }, { 61, 0 }, { 1, 3 }, { 1, 203 }, { 9, 255 }, { 1, 212 }, { 1, 148 }, { 1, 83 }, { 1, 20 }, { 26, 0 }, { 1, 1 }, { 38, 0 }, { 1, 105 }, { 6, 255 }, { 1, 216 }, { 1, 151 }, { 1, 86 }, { 1, 22 }, { 28, 0 }, { 1, 5 }, { 1, 21 }, { 1, 4 }, { 38, 0 }, { 1, 183 }, { 2, 255 }, { 1, 220 }, { 1, 155 }, { 1, 90 }, { 1, 26 }, { 30, 0 }, { 1, 3 }, { 1, 32 }, { 1, 33 }, { 1, 1 }, { 39, 0 }, { 1, 33 }, { 1, 74 }, { 1, 29 }, { 32, 0 }, { 1, 1 }, { 1, 29 }, { 1, 56 }, { 1, 22 }, { 75, 0 }, { 1, 17 }, { 1, 53 }, { 1, 50 }, { 1, 10 }, { 74, 0 }, { 1, 6 }, { 1, 42 }, { 1, 60 }, { 1, 28 }, { 75, 0 }, { 1, 28 }, { 1, 59 }, { 1, 45 }, { 1, 8 }, { 74, 0 }, { 1, 6 }, { 1, 46 }, { 1, 55 }, { 1, 22 }, { 75, 0 }, { 1, 16 }, { 1, 49 }, { 1, 24 }, { 1, 1 }, { 75, 0 }, { 2, 23 }, { 77, 0 }, { 1, 8 }, { 1391, 0 } }, 
	LogoRLEFrame{ { 1411, 0 }, { 1, 8 }, { 1, 2 }, { 77, 0 }, { 1, 11 }, { 1, 33 }, { 77, 0 }, { 1, 13 }, { 1, 55 }, { 1, 5 }, { 55, 0 }, { 1, 6 }, { 20, 0 }, { 1, 7 }, { 1, 57 }, { 1, 24 }, { 56, 0 }, { 1, 21 }, { 19, 0 }, { 1, 3 }, { 1, 51 }, { 1, 39 }, { 56, 0 }, { 1, 27 }, { 1, 9 }, { 19, 0 }, { 1, 44 }, { 1, 50 }, { 1, 1 }, { 55, 0 }, { 1, 7 }, { 1, 43 }, { 15, 0 }, { 1, 8 }, { 3, 0 }, { 1, 26 }, { 1, 57 }, { 1, 7 }, { 56, 0 }, { 1, 45 }, { 1, 20 }, { 13, 0 }, { 1, 15 }, { 1, 200 }, { 1, 249 }, { 1, 54 }, { 1, 0 }, { 1, 7 }, { 1, 52 }, { 1, 9 }, { 56, 0 }, { 1, 21 }, { 1, 55 }, { 14, 0 }, { 1, 158 }, { 2, 255 }, { 1, 150 }, { 1, 0 }, { 1, 36 }, { 1, 7 }, { 56, 0 }, { 1, 1 }, { 1, 54 }, { 1, 24 }, { 13, 0 }, { 1, 55 }, { 3, 255 }, { 1, 216 }, { 1, 13 }, { 1, 5 }, { 57, 0 }, { 1, 27 }, { 1, 53 }, { 13, 0 }, { 1, 2 }, { 1, 203 }, { 4, 255 }, { 1, 25 }, { 57, 0 }, { 1, 3 }, { 1, 58 }, { 1, 22 }, { 13, 0 }, { 1, 100 }, { 5, 255 }, { 1, 60 }, { 57, 0 }, { 1, 33 }, { 1, 51 }, { 13, 0 }, { 1, 17 }, { 1, 235 }, { 5, 255 }, { 1, 88 }, { 56, 0 }, { 1, 6 }, { 1, 60 }, { 1, 19 }, { 13, 0 }, { 1, 147 }, { 6, 255 }, { 1, 109 }, { 56, 0 }, { 1, 32 }, { 1, 45 }, { 13, 0 }, { 1, 46 }, { 1, 252 }, { 6, 255 }, { 1, 54 }, { 55, 0 }, { 1, 1 }, { 1, 55 }, { 1, 8 }, { 13, 0 }, { 1, 193 }, { 6, 255 }, { 1, 211 }, { 1, 1 }, { 55, 0 }, { 1, 23 }, { 1, 29 }, { 13, 0 }, { 1, 90 }, { 6, 255 }, { 1, 251 }, { 1, 50 }, { 56, 0 }, { 1, 36 }, { 13, 0 }, { 1, 12 }, { 1, 229 }, { 6, 255 }, { 1, 106 }, { 56, 0 }, { 1, 11 }, { 1, 7 }, { 13, 0 }, { 1, 137 }, { 6, 255 }, { 1, 148 }, { 57, 0 }, { 1, 2 }, { 13, 0 }, { 1, 38 }, { 1, 249 }, { 5, 255 }, { 1, 187 }, { 1, 4 }, { 71, 0 }, { 1, 183 }, { 5, 255 }, { 1, 217 }, { 1, 16 }, { 2, 0 }, { 1, 4 }, { 1, 109 }, { 1, 172 }, { 1, 145 }, { 1, 26 }, { 64, 0 }, { 1, 79 }, { 5, 255 }, { 1, 238 }, { 1, 37 }, { 2, 0 }, { 1, 22 }, { 1, 199 }, { 3, 255 }, { 1, 194 }, { 63, 0 }, { 1, 8 }, { 1, 222 }, { 4, 255 }, { 1, 251 }, { 1, 67 }, { 2, 0 }, { 1, 41 }, { 1, 222 }, { 4, 255 }, { 1, 253 }, { 1, 22 }, { 62, 0 }, { 1, 126 }, { 5, 255 }, { 1, 106 }, { 2, 0 }, { 1, 67 }, { 1, 240 }, { 6, 255 }, { 1, 100 }, { 61, 0 }, { 1, 31 }, { 1, 246 }, { 4, 255 }, { 1, 148 }, { 2, 0 }, { 1, 99 }, { 1, 250 }, { 7, 255 }, { 1, 181 }, { 61, 0 }, { 1, 173 }, { 4, 255 }, { 1, 187 }, { 1, 4 }, { 1, 1 }, { 1, 136 }, { 9, 255 }, { 1, 249 }, { 1, 13 }, { 59, 0 }, { 1, 69 }, { 4, 255 }, { 1, 217 }, { 1, 16 }, { 1, 9 }, { 1, 171 }, { 11, 255 }, { 1, 87 }, { 58, 0 }, { 1, 5 }, { 1, 215 }, { 3, 255 }, { 1, 238 }, { 1, 37 }, { 1, 22 }, { 1, 200 }, { 12, 255 }, { 1, 168 }, { 58, 0 }, { 1, 115 }, { 3, 255 }, { 1, 251 }, { 1, 67 }, { 1, 42 }, { 1, 223 }, { 13, 255 }, { 1, 242 }, { 1, 7 }, { 56, 0 }, { 1, 24 }, { 1, 242 }, { 3, 255 }, { 1, 173 }, { 1, 72 }, { 1, 240 }, { 15, 255 }, { 1, 74 }, { 56, 0 }, { 1, 162 }, { 22, 255 }, { 1, 144 }, { 55, 0 }, { 1, 59 }, { 23, 255 }, { 1, 164 }, { 54, 0 }, { 1, 2 }, { 1, 206 }, { 22, 255 }, { 1, 251 }, { 1, 54 }, { 54, 0 }, { 1, 105 }, { 22, 255 }, { 1, 236 }, { 1, 69 }, { 54, 0 }, { 1, 19 }, { 1, 237 }, { 19, 255 }, { 1, 224 }, { 1, 168 }, { 1, 98 }, { 1, 9 }, { 16, 0 }, { 1, 12 }, { 1, 4 }, { 37, 0 }, { 1, 151 }, { 15, 255 }, { 1, 243 }, { 1, 190 }, { 1, 134 }, { 1, 79 }, { 1, 24 }, { 18, 0 }, { 1, 15 }, { 1, 34 }, { 1, 7 }, { 37, 0 }, { 1, 49 }, { 1, 253 }, { 10, 255 }, { 1, 253 }, { 1, 211 }, { 1, 155 }, { 1, 100 }, { 1, 45 }, { 1, 2 }, { 20, 0 }, { 1, 13 }, { 1, 46 }, { 1, 37 }, { 1, 1 }, { 38, 0 }, { 1, 179 }, { 7, 255 }, { 1, 232 }, { 1, 177 }, { 1, 121 }, { 1, 66 }, { 1, 13 }, { 4, 0 }, { 1, 2 }, { 1, 8 }, { 17, 0 }, { 1, 5 }, { 1, 39 }, { 1, 59 }, { 1, 23 }, { 40, 0 }, { 1, 223 }, { 2, 255 }, { 1, 248 }, { 1, 198 }, { 1, 142 }, { 1, 87 }, { 1, 32 }, { 8, 0 }, { 1, 9 }, { 1, 20 }, { 17, 0 }, { 1, 26 }, { 1, 59 }, { 1, 43 }, { 1, 6 }, { 41, 0 }, { 1, 39 }, { 1, 88 }, { 1, 53 }, { 1, 6 }, { 11, 0 }, { 1, 21 }, { 1, 25 }, { 16, 0 }, { 1, 13 }, { 1, 50 }, { 1, 56 }, { 1, 20 }, { 56, 0 }, { 1, 5 }, { 1, 32 }, { 1, 26 }, { 16, 0 }, { 1, 32 }, { 1, 62 }, { 1, 38 }, { 1, 3 }, { 56, 0 }, { 1, 10 }, { 1, 38 }, { 1, 24 }, { 15, 0 }, { 1, 6 }, { 1, 47 }, { 1, 41 }, { 1, 11 }, { 57, 0 }, { 1, 17 }, { 1, 40 }, { 1, 18 }, { 15, 0 }, { 1, 17 }, { 1, 35 }, { 1, 10 }, { 58, 0 }, { 1, 24 }, { 1, 40 }, { 1, 12 }, { 15, 0 }, { 1, 11 }, { 1, 9 }, { 58, 0 }, { 1, 1 }, { 1, 31 }, { 1, 37 }, { 1, 8 }, { 75, 0 }, { 1, 2 }, { 1, 32 }, { 1, 31 }, { 1, 4 }, { 75, 0 }, { 1, 2 }, { 1, 31 }, { 1, 20 }, { 76, 0 }, { 1, 2 }, { 1, 25 }, { 1, 9 }, { 76, 0 }, { 1, 1 }, { 1, 11 }, { 1171, 0 } }, 
	LogoRLEFrame{ { 1174, 0 }, { 1, 1 }, { 78, 0 }, { 1, 29 }, { 77, 0 }, { 1, 1 }, { 1, 43 }, { 1, 17 }, { 55, 0 }, { 1, 11 }, { 21, 0 }, { 1, 43 }, { 1, 42 }, { 55, 0 }, { 1, 7 }, { 1, 18 }, { 20, 0 }, { 1, 33 }, { 1, 54 }, { 1, 4 }, { 55, 0 }, { 1, 38 }, { 1, 1 }, { 19, 0 }, { 1, 24 }, { 1, 60 }, { 1, 12 }, { 55, 0 }, { 1, 21 }, { 1, 33 }, { 19, 0 }, { 1, 11 }, { 1, 61 }, { 1, 23 }, { 55, 0 }, { 1, 4 }, { 1, 57 }, { 1, 8 }, { 19, 0 }, { 1, 50 }, { 1, 29 }, { 56, 0 }, { 1, 36 }, { 1, 42 }, { 19, 0 }, { 1, 29 }, { 1, 25 }, { 56, 0 }, { 1, 9 }, { 1, 61 }, { 1, 10 }, { 13, 0 }, { 1, 44 }, { 1, 19 }, { 3, 0 }, { 1, 9 }, { 1, 20 }, { 57, 0 }, { 1, 42 }, { 1, 40 }, { 13, 0 }, { 1, 125 }, { 1, 255 }, { 1, 232 }, { 1, 19 }, { 2, 0 }, { 1, 5 }, { 57, 0 }, { 1, 14 }, { 1, 61 }, { 1, 9 }, { 12, 0 }, { 1, 38 }, { 1, 251 }, { 2, 255 }, { 1, 107 }, { 60, 0 }, { 1, 48 }, { 1, 38 }, { 13, 0 }, { 1, 170 }, { 3, 255 }, { 1, 191 }, { 59, 0 }, { 1, 17 }, { 1, 60 }, { 1, 7 }, { 12, 0 }, { 1, 51 }, { 4, 255 }, { 1, 252 }, { 1, 22 }, { 58, 0 }, { 1, 45 }, { 1, 30 }, { 13, 0 }, { 1, 186 }, { 5, 255 }, { 1, 85 }, { 57, 0 }, { 1, 9 }, { 1, 52 }, { 1, 1 }, { 12, 0 }, { 1, 66 }, { 6, 255 }, { 1, 132 }, { 57, 0 }, { 1, 35 }, { 1, 14 }, { 13, 0 }, { 1, 202 }, { 6, 255 }, { 1, 180 }, { 56, 0 }, { 1, 3 }, { 1, 29 }, { 13, 0 }, { 1, 83 }, { 7, 255 }, { 1, 210 }, { 56, 0 }, { 1, 14 }, { 13, 0 }, { 1, 3 }, { 1, 216 }, { 7, 255 }, { 1, 167 }, { 70, 0 }, { 1, 99 }, { 8, 255 }, { 1, 86 }, { 69, 0 }, { 1, 8 }, { 1, 227 }, { 7, 255 }, { 1, 192 }, { 70, 0 }, { 1, 116 }, { 7, 255 }, { 1, 235 }, { 1, 31 }, { 69, 0 }, { 1, 15 }, { 1, 237 }, { 6, 255 }, { 1, 251 }, { 1, 66 }, { 70, 0 }, { 1, 132 }, { 7, 255 }, { 1, 112 }, { 70, 0 }, { 1, 24 }, { 1, 244 }, { 6, 255 }, { 1, 163 }, { 4, 0 }, { 1, 7 }, { 1, 45 }, { 1, 27 }, { 64, 0 }, { 1, 149 }, { 6, 255 }, { 1, 205 }, { 1, 9 }, { 3, 0 }, { 1, 81 }, { 1, 231 }, { 2, 255 }, { 1, 173 }, { 1, 2 }, { 61, 0 }, { 1, 35 }, { 1, 250 }, { 5, 255 }, { 1, 234 }, { 1, 30 }, { 2, 0 }, { 1, 1 }, { 1, 129 }, { 5, 255 }, { 1, 76 }, { 61, 0 }, { 1, 165 }, { 5, 255 }, { 1, 251 }, { 1, 64 }, { 2, 0 }, { 1, 11 }, { 1, 173 }, { 6, 255 }, { 1, 184 }, { 60, 0 }, { 1, 47 }, { 6, 255 }, { 1, 110 }, { 2, 0 }, { 1, 31 }, { 1, 208 }, { 8, 255 }, { 1, 37 }, { 59, 0 }, { 1, 181 }, { 5, 255 }, { 1, 161 }, { 2, 0 }, { 1, 60 }, { 1, 234 }, { 9, 255 }, { 1, 143 }, { 58, 0 }, { 1, 62 }, { 5, 255 }, { 1, 204 }, { 1, 8 }, { 1, 0 }, { 1, 99 }, { 1, 249 }, { 10, 255 }, { 1, 240 }, { 1, 11 }, { 57, 0 }, { 1, 198 }, { 4, 255 }, { 1, 233 }, { 1, 29 }, { 1, 3 }, { 1, 145 }, { 13, 255 }, { 1, 102 }, { 56, 0 }, { 1, 78 }, { 4, 255 }, { 1, 251 }, { 1, 63 }, { 1, 17 }, { 1, 186 }, { 14, 255 }, { 1, 210 }, { 55, 0 }, { 1, 2 }, { 1, 212 }, { 4, 255 }, { 1, 209 }, { 1, 49 }, { 1, 218 }, { 16, 255 }, { 1, 61 }, { 54, 0 }, { 1, 95 }, { 24, 255 }, { 1, 169 }, { 53, 0 }, { 1, 7 }, { 1, 224 }, { 24, 255 }, { 1, 250 }, { 1, 12 }, { 14, 0 }, { 1, 3 }, { 1, 2 }, { 36, 0 }, { 1, 111 }, { 25, 255 }, { 1, 237 }, { 1, 8 }, { 12, 0 }, { 1, 3 }, { 1, 26 }, { 1, 13 }, { 36, 0 }, { 1, 13 }, { 1, 234 }, { 25, 255 }, { 1, 109 }, { 11, 0 }, { 1, 2 }, { 1, 29 }, { 1, 45 }, { 1, 7 }, { 37, 0 }, { 1, 128 }, { 24, 255 }, { 1, 222 }, { 1, 92 }, { 11, 0 }, { 1, 24 }, { 1, 57 }, { 1, 37 }, { 1, 1 }, { 37, 0 }, { 1, 21 }, { 1, 243 }, { 19, 255 }, { 1, 240 }, { 1, 195 }, { 1, 150 }, { 1, 105 }, { 1, 57 }, { 1, 1 }, { 10, 0 }, { 1, 11 }, { 1, 48 }, { 1, 55 }, { 1, 18 }, { 39, 0 }, { 1, 144 }, { 14, 255 }, { 1, 250 }, { 1, 211 }, { 1, 183 }, { 1, 139 }, { 1, 74 }, { 1, 29 }, { 14, 0 }, { 1, 3 }, { 1, 35 }, { 1, 61 }, { 1, 36 }, { 1, 3 }, { 39, 0 }, { 1, 2 }, { 1, 239 }, { 9, 255 }, { 1, 225 }, { 1, 180 }, { 1, 134 }, { 1, 89 }, { 1, 44 }, { 1, 23 }, { 1, 59 }, { 1, 31 }, { 16, 0 }, { 1, 18 }, { 1, 56 }, { 1, 52 }, { 1, 14 }, { 41, 0 }, { 1, 3 }, { 1, 231 }, { 3, 255 }, { 1, 240 }, { 1, 195 }, { 1, 149 }, { 1, 104 }, { 1, 59 }, { 1, 15 }, { 4, 0 }, { 1, 31 }, { 1, 61 }, { 1, 22 }, { 15, 0 }, { 1, 1 }, { 1, 34 }, { 1, 56 }, { 1, 27 }, { 1, 1 }, { 43, 0 }, { 1, 30 }, { 1, 92 }, { 1, 74 }, { 1, 29 }, { 8, 0 }, { 1, 2 }, { 1, 42 }, { 1, 58 }, { 1, 15 }, { 15, 0 }, { 1, 7 }, { 1, 42 }, { 1, 26 }, { 1, 1 }, { 56, 0 }, { 1, 2 }, { 1, 47 }, { 1, 52 }, { 1, 9 }, { 15, 0 }, { 1, 12 }, { 1, 21 }, { 1, 1 }, { 57, 0 }, { 1, 2 }, { 1, 47 }, { 1, 38 }, { 1, 2 }, { 16, 0 }, { 1, 1 }, { 58, 0 }, { 1, 3 }, { 1, 44 }, { 1, 21 }, { 76, 0 }, { 1, 3 }, { 1, 27 }, { 1, 5 }, { 77, 0 }, { 1, 4 }, { 1407, 0 } }, 
	LogoRLEFrame{ { 1015, 0 }, { 1, 6 }, { 1, 1 }, { 54, 0 }, { 1, 1 }, { 22, 0 }, { 1, 9 }, { 1, 14 }, { 55, 0 }, { 1, 14 }, { 21, 0 }, { 1, 10 }, { 1, 26 }, { 1, 1 }, { 54, 0 }, { 1, 19 }, { 1, 9 }, { 20, 0 }, { 1, 7 }, { 1, 29 }, { 1, 8 }, { 54, 0 }, { 1, 3 }, { 1, 39 }, { 20, 0 }, { 1, 3 }, { 1, 28 }, { 1, 14 }, { 55, 0 }, { 1, 38 }, { 1, 20 }, { 19, 0 }, { 1, 1 }, { 1, 25 }, { 1, 21 }, { 55, 0 }, { 1, 15 }, { 1, 56 }, { 20, 0 }, { 1, 17 }, { 1, 24 }, { 1, 1 }, { 55, 0 }, { 1, 50 }, { 1, 28 }, { 19, 0 }, { 1, 6 }, { 1, 23 }, { 1, 1 }, { 55, 0 }, { 1, 22 }, { 1, 56 }, { 1, 2 }, { 19, 0 }, { 1, 17 }, { 1, 1 }, { 55, 0 }, { 1, 1 }, { 1, 55 }, { 1, 26 }, { 19, 0 }, { 1, 6 }, { 57, 0 }, { 1, 29 }, { 1, 55 }, { 1, 1 }, { 76, 0 }, { 1, 4 }, { 1, 58 }, { 1, 23 }, { 11, 0 }, { 1, 53 }, { 1, 218 }, { 1, 196 }, { 1, 20 }, { 62, 0 }, { 1, 31 }, { 1, 51 }, { 11, 0 }, { 1, 2 }, { 1, 217 }, { 2, 255 }, { 1, 155 }, { 61, 0 }, { 1, 1 }, { 1, 56 }, { 1, 14 }, { 11, 0 }, { 1, 87 }, { 3, 255 }, { 1, 243 }, { 1, 12 }, { 60, 0 }, { 1, 22 }, { 1, 37 }, { 12, 0 }, { 1, 207 }, { 4, 255 }, { 1, 99 }, { 60, 0 }, { 1, 42 }, { 1, 4 }, { 11, 0 }, { 1, 73 }, { 5, 255 }, { 1, 198 }, { 59, 0 }, { 1, 13 }, { 1, 15 }, { 12, 0 }, { 1, 194 }, { 6, 255 }, { 1, 20 }, { 58, 0 }, { 1, 10 }, { 12, 0 }, { 1, 59 }, { 7, 255 }, { 1, 82 }, { 71, 0 }, { 1, 180 }, { 7, 255 }, { 1, 146 }, { 70, 0 }, { 1, 47 }, { 8, 255 }, { 1, 206 }, { 70, 0 }, { 1, 167 }, { 8, 255 }, { 1, 198 }, { 69, 0 }, { 1, 35 }, { 1, 252 }, { 8, 255 }, { 1, 162 }, { 69, 0 }, { 1, 153 }, { 9, 255 }, { 1, 53 }, { 68, 0 }, { 1, 25 }, { 1, 248 }, { 8, 255 }, { 1, 167 }, { 69, 0 }, { 1, 139 }, { 8, 255 }, { 1, 218 }, { 1, 14 }, { 68, 0 }, { 1, 17 }, { 1, 243 }, { 7, 255 }, { 1, 246 }, { 1, 46 }, { 69, 0 }, { 1, 126 }, { 8, 255 }, { 1, 95 }, { 69, 0 }, { 1, 11 }, { 1, 236 }, { 7, 255 }, { 1, 154 }, { 5, 0 }, { 1, 20 }, { 1, 50 }, { 1, 15 }, { 62, 0 }, { 1, 112 }, { 7, 255 }, { 1, 204 }, { 1, 7 }, { 3, 0 }, { 1, 8 }, { 1, 144 }, { 1, 252 }, { 1, 255 }, { 1, 253 }, { 1, 148 }, { 60, 0 }, { 1, 6 }, { 1, 227 }, { 6, 255 }, { 1, 238 }, { 1, 33 }, { 3, 0 }, { 1, 31 }, { 1, 204 }, { 5, 255 }, { 1, 83 }, { 59, 0 }, { 1, 98 }, { 7, 255 }, { 1, 76 }, { 3, 0 }, { 1, 69 }, { 1, 236 }, { 6, 255 }, { 1, 215 }, { 1, 3 }, { 57, 0 }, { 1, 2 }, { 1, 217 }, { 6, 255 }, { 1, 133 }, { 2, 0 }, { 1, 1 }, { 1, 121 }, { 1, 253 }, { 8, 255 }, { 1, 97 }, { 57, 0 }, { 1, 85 }, { 6, 255 }, { 1, 189 }, { 1, 3 }, { 1, 0 }, { 1, 14 }, { 1, 175 }, { 10, 255 }, { 1, 225 }, { 1, 7 }, { 56, 0 }, { 1, 205 }, { 5, 255 }, { 1, 229 }, { 1, 22 }, { 1, 0 }, { 1, 42 }, { 1, 217 }, { 12, 255 }, { 1, 111 }, { 55, 0 }, { 1, 71 }, { 5, 255 }, { 1, 251 }, { 1, 59 }, { 1, 0 }, { 1, 86 }, { 1, 243 }, { 13, 255 }, { 1, 234 }, { 1, 12 }, { 54, 0 }, { 1, 192 }, { 5, 255 }, { 1, 150 }, { 1, 4 }, { 1, 141 }, { 16, 255 }, { 1, 125 }, { 53, 0 }, { 1, 58 }, { 6, 255 }, { 1, 228 }, { 1, 202 }, { 17, 255 }, { 1, 241 }, { 1, 19 }, { 52, 0 }, { 1, 179 }, { 26, 255 }, { 1, 140 }, { 51, 0 }, { 1, 45 }, { 27, 255 }, { 1, 245 }, { 1, 13 }, { 50, 0 }, { 1, 165 }, { 28, 255 }, { 1, 25 }, { 49, 0 }, { 1, 34 }, { 1, 252 }, { 27, 255 }, { 1, 179 }, { 50, 0 }, { 1, 151 }, { 27, 255 }, { 1, 170 }, { 1, 11 }, { 49, 0 }, { 1, 24 }, { 1, 248 }, { 21, 255 }, { 1, 248 }, { 1, 214 }, { 1, 178 }, { 1, 142 }, { 1, 105 }, { 1, 31 }, { 51, 0 }, { 1, 111 }, { 15, 255 }, { 1, 248 }, { 1, 219 }, { 1, 176 }, { 1, 139 }, { 1, 103 }, { 1, 67 }, { 1, 31 }, { 1, 2 }, { 56, 0 }, { 1, 127 }, { 8, 255 }, { 1, 245 }, { 1, 209 }, { 1, 173 }, { 1, 137 }, { 1, 101 }, { 1, 66 }, { 1, 69 }, { 1, 29 }, { 63, 0 }, { 1, 24 }, { 1, 184 }, { 1, 227 }, { 1, 207 }, { 1, 171 }, { 1, 134 }, { 1, 98 }, { 1, 62 }, { 1, 26 }, { 4, 0 }, { 1, 3 }, { 1, 36 }, { 1, 12 }, { 76, 0 }, { 1, 1 }, { 1, 15 }, { 1722, 0 } }, 
	LogoRLEFrame{ { 833, 0 }, { 1, 1 }, { 78, 0 }, { 1, 1 }, { 1, 7 }, { 78, 0 }, { 1, 15 }, { 78, 0 }, { 1, 7 }, { 1, 16 }, { 78, 0 }, { 1, 25 }, { 1, 3 }, { 77, 0 }, { 1, 15 }, { 1, 22 }, { 77, 0 }, { 1, 3 }, { 1, 29 }, { 1, 6 }, { 77, 0 }, { 1, 18 }, { 1, 21 }, { 77, 0 }, { 1, 5 }, { 1, 30 }, { 1, 5 }, { 77, 0 }, { 1, 21 }, { 1, 20 }, { 77, 0 }, { 1, 7 }, { 1, 30 }, { 1, 4 }, { 77, 0 }, { 1, 22 }, { 1, 18 }, { 77, 0 }, { 1, 4 }, { 1, 27 }, { 1, 1 }, { 9, 0 }, { 1, 104 }, { 1, 242 }, { 1, 191 }, { 1, 15 }, { 64, 0 }, { 1, 17 }, { 1, 10 }, { 9, 0 }, { 1, 23 }, { 1, 249 }, { 2, 255 }, { 1, 156 }, { 63, 0 }, { 1, 1 }, { 1, 19 }, { 10, 0 }, { 1, 126 }, { 3, 255 }, { 1, 248 }, { 1, 22 }, { 62, 0 }, { 1, 9 }, { 1, 2 }, { 9, 0 }, { 1, 4 }, { 1, 229 }, { 4, 255 }, { 1, 125 }, { 62, 0 }, { 1, 3 }, { 10, 0 }, { 1, 86 }, { 5, 255 }, { 1, 230 }, { 1, 5 }, { 72, 0 }, { 1, 194 }, { 6, 255 }, { 1, 82 }, { 71, 0 }, { 1, 47 }, { 7, 255 }, { 1, 158 }, { 71, 0 }, { 1, 155 }, { 7, 255 }, { 1, 233 }, { 1, 1 }, { 69, 0 }, { 1, 17 }, { 1, 246 }, { 8, 255 }, { 1, 53 }, { 69, 0 }, { 1, 115 }, { 9, 255 }, { 1, 117 }, { 68, 0 }, { 1, 2 }, { 1, 221 }, { 9, 255 }, { 1, 114 }, { 68, 0 }, { 1, 76 }, { 10, 255 }, { 1, 90 }, { 68, 0 }, { 1, 184 }, { 9, 255 }, { 1, 236 }, { 1, 9 }, { 67, 0 }, { 1, 38 }, { 10, 255 }, { 1, 119 }, { 68, 0 }, { 1, 145 }, { 9, 255 }, { 1, 190 }, { 1, 2 }, { 51, 0 }, { 1, 7 }, { 1, 4 }, { 14, 0 }, { 1, 12 }, { 1, 241 }, { 8, 255 }, { 1, 234 }, { 1, 25 }, { 52, 0 }, { 1, 27 }, { 15, 0 }, { 1, 105 }, { 9, 255 }, { 1, 72 }, { 52, 0 }, { 1, 12 }, { 1, 29 }, { 15, 0 }, { 1, 213 }, { 8, 255 }, { 1, 138 }, { 53, 0 }, { 1, 31 }, { 1, 18 }, { 14, 0 }, { 1, 66 }, { 8, 255 }, { 1, 200 }, { 1, 5 }, { 4, 0 }, { 1, 27 }, { 1, 158 }, { 1, 209 }, { 1, 192 }, { 1, 113 }, { 1, 5 }, { 42, 0 }, { 1, 6 }, { 1, 41 }, { 1, 3 }, { 14, 0 }, { 1, 174 }, { 7, 255 }, { 1, 240 }, { 1, 32 }, { 4, 0 }, { 1, 80 }, { 1, 239 }, { 4, 255 }, { 1, 163 }, { 42, 0 }, { 1, 23 }, { 1, 29 }, { 14, 0 }, { 1, 30 }, { 1, 252 }, { 7, 255 }, { 1, 84 }, { 3, 0 }, { 1, 5 }, { 1, 142 }, { 7, 255 }, { 1, 68 }, { 40, 0 }, { 1, 1 }, { 1, 39 }, { 1, 14 }, { 14, 0 }, { 1, 134 }, { 7, 255 }, { 1, 150 }, { 3, 0 }, { 1, 31 }, { 1, 200 }, { 8, 255 }, { 1, 219 }, { 1, 8 }, { 2, 0 }, { 1, 4 }, { 36, 0 }, { 1, 10 }, { 1, 37 }, { 1, 1 }, { 13, 0 }, { 1, 8 }, { 1, 235 }, { 6, 255 }, { 1, 209 }, { 1, 8 }, { 2, 0 }, { 1, 77 }, { 1, 238 }, { 10, 255 }, { 1, 131 }, { 1, 8 }, { 1, 25 }, { 37, 0 }, { 1, 22 }, { 1, 15 }, { 14, 0 }, { 1, 95 }, { 6, 255 }, { 1, 244 }, { 1, 40 }, { 1, 0 }, { 1, 4 }, { 1, 139 }, { 12, 255 }, { 1, 250 }, { 1, 72 }, { 38, 0 }, { 1, 26 }, { 15, 0 }, { 1, 204 }, { 6, 255 }, { 1, 96 }, { 1, 0 }, { 1, 29 }, { 1, 197 }, { 14, 255 }, { 1, 195 }, { 1, 1 }, { 36, 0 }, { 1, 3 }, { 1, 11 }, { 14, 0 }, { 1, 56 }, { 6, 255 }, { 1, 228 }, { 1, 0 }, { 1, 74 }, { 1, 236 }, { 16, 255 }, { 1, 99 }, { 36, 0 }, { 1, 4 }, { 15, 0 }, { 1, 164 }, { 7, 255 }, { 1, 196 }, { 18, 255 }, { 1, 238 }, { 1, 22 }, { 50, 0 }, { 1, 23 }, { 1, 249 }, { 27, 255 }, { 1, 163 }, { 50, 0 }, { 1, 124 }, { 29, 255 }, { 1, 61 }, { 48, 0 }, { 1, 4 }, { 1, 228 }, { 29, 255 }, { 1, 154 }, { 48, 0 }, { 1, 85 }, { 30, 255 }, { 1, 112 }, { 48, 0 }, { 1, 193 }, { 29, 255 }, { 1, 204 }, { 1, 17 }, { 47, 0 }, { 1, 45 }, { 27, 255 }, { 1, 247 }, { 1, 205 }, { 1, 119 }, { 1, 11 }, { 48, 0 }, { 1, 107 }, { 19, 255 }, { 1, 231 }, { 1, 201 }, { 1, 171 }, { 1, 141 }, { 1, 111 }, { 1, 81 }, { 1, 51 }, { 1, 21 }, { 52, 0 }, { 1, 84 }, { 10, 255 }, { 1, 245 }, { 1, 215 }, { 1, 185 }, { 1, 155 }, { 1, 125 }, { 1, 95 }, { 1, 65 }, { 1, 35 }, { 1, 7 }, { 60, 0 }, { 1, 3 }, { 1, 140 }, { 1, 224 }, { 1, 228 }, { 1, 198 }, { 1, 168 }, { 1, 138 }, { 1, 108 }, { 1, 78 }, { 1, 48 }, { 1, 19 }, { 1805, 0 } }, 
	LogoRLEFrame{ { 1717, 0 }, { 1, 74 }, { 1, 216 }, { 1, 178 }, { 1, 22 }, { 75, 0 }, { 1, 6 }, { 1, 234 }, { 2, 255 }, { 1, 190 }, { 75, 0 }, { 1, 84 }, { 4, 255 }, { 1, 62 }, { 74, 0 }, { 1, 182 }, { 4, 255 }, { 1, 182 }, { 73, 0 }, { 1, 28 }, { 1, 253 }, { 5, 255 }, { 1, 48 }, { 72, 0 }, { 1, 123 }, { 6, 255 }, { 1, 165 }, { 71, 0 }, { 1, 1 }, { 1, 220 }, { 6, 255 }, { 1, 247 }, { 1, 12 }, { 70, 0 }, { 1, 63 }, { 8, 255 }, { 1, 88 }, { 70, 0 }, { 1, 161 }, { 8, 255 }, { 1, 174 }, { 69, 0 }, { 1, 14 }, { 1, 245 }, { 8, 255 }, { 1, 247 }, { 1, 12 }, { 68, 0 }, { 1, 101 }, { 10, 255 }, { 1, 58 }, { 68, 0 }, { 1, 200 }, { 10, 255 }, { 1, 58 }, { 52, 0 }, { 1, 17 }, { 14, 0 }, { 1, 42 }, { 11, 255 }, { 1, 33 }, { 51, 0 }, { 1, 8 }, { 1, 31 }, { 14, 0 }, { 1, 140 }, { 10, 255 }, { 1, 191 }, { 52, 0 }, { 1, 42 }, { 1, 17 }, { 13, 0 }, { 1, 5 }, { 1, 233 }, { 10, 255 }, { 1, 68 }, { 51, 0 }, { 1, 9 }, { 1, 61 }, { 1, 3 }, { 13, 0 }, { 1, 80 }, { 10, 255 }, { 1, 142 }, { 52, 0 }, { 1, 34 }, { 1, 42 }, { 14, 0 }, { 1, 178 }, { 9, 255 }, { 1, 209 }, { 1, 7 }, { 51, 0 }, { 1, 1 }, { 1, 58 }, { 1, 19 }, { 13, 0 }, { 1, 25 }, { 1, 252 }, { 8, 255 }, { 1, 247 }, { 1, 43 }, { 52, 0 }, { 1, 22 }, { 1, 58 }, { 1, 1 }, { 13, 0 }, { 1, 118 }, { 9, 255 }, { 1, 108 }, { 53, 0 }, { 1, 41 }, { 1, 33 }, { 14, 0 }, { 1, 216 }, { 8, 255 }, { 1, 182 }, { 5, 0 }, { 1, 2 }, { 1, 102 }, { 1, 171 }, { 1, 174 }, { 1, 118 }, { 1, 13 }, { 7, 0 }, { 1, 4 }, { 1, 15 }, { 34, 0 }, { 1, 53 }, { 1, 3 }, { 13, 0 }, { 1, 59 }, { 8, 255 }, { 1, 233 }, { 1, 23 }, { 4, 0 }, { 1, 29 }, { 1, 194 }, { 4, 255 }, { 1, 226 }, { 1, 20 }, { 5, 0 }, { 1, 15 }, { 1, 32 }, { 34, 0 }, { 1, 13 }, { 1, 28 }, { 14, 0 }, { 1, 157 }, { 8, 255 }, { 1, 75 }, { 4, 0 }, { 1, 80 }, { 1, 237 }, { 6, 255 }, { 1, 175 }, { 3, 0 }, { 1, 1 }, { 1, 34 }, { 1, 39 }, { 35, 0 }, { 1, 22 }, { 1, 1 }, { 13, 0 }, { 1, 12 }, { 1, 243 }, { 7, 255 }, { 1, 148 }, { 3, 0 }, { 1, 8 }, { 1, 148 }, { 9, 255 }, { 1, 99 }, { 1, 0 }, { 1, 8 }, { 1, 50 }, { 1, 40 }, { 36, 0 }, { 1, 7 }, { 14, 0 }, { 1, 97 }, { 7, 255 }, { 1, 214 }, { 1, 9 }, { 2, 0 }, { 1, 40 }, { 1, 209 }, { 10, 255 }, { 1, 244 }, { 1, 48 }, { 1, 58 }, { 1, 35 }, { 52, 0 }, { 1, 195 }, { 6, 255 }, { 1, 249 }, { 1, 48 }, { 2, 0 }, { 1, 98 }, { 1, 245 }, { 12, 255 }, { 1, 212 }, { 1, 29 }, { 52, 0 }, { 1, 38 }, { 7, 255 }, { 1, 117 }, { 1, 0 }, { 1, 14 }, { 1, 167 }, { 15, 255 }, { 1, 128 }, { 52, 0 }, { 1, 135 }, { 7, 255 }, { 1, 58 }, { 1, 54 }, { 1, 221 }, { 16, 255 }, { 1, 252 }, { 1, 56 }, { 50, 0 }, { 1, 3 }, { 1, 230 }, { 7, 255 }, { 1, 242 }, { 1, 252 }, { 18, 255 }, { 1, 220 }, { 1, 12 }, { 49, 0 }, { 1, 76 }, { 29, 255 }, { 1, 156 }, { 49, 0 }, { 1, 174 }, { 30, 255 }, { 1, 75 }, { 47, 0 }, { 1, 22 }, { 1, 250 }, { 30, 255 }, { 1, 189 }, { 47, 0 }, { 1, 114 }, { 31, 255 }, { 1, 180 }, { 47, 0 }, { 1, 212 }, { 30, 255 }, { 1, 250 }, { 1, 73 }, { 46, 0 }, { 1, 48 }, { 29, 255 }, { 1, 252 }, { 1, 189 }, { 1, 67 }, { 47, 0 }, { 1, 82 }, { 20, 255 }, { 1, 251 }, { 1, 226 }, { 1, 200 }, { 1, 174 }, { 1, 147 }, { 1, 121 }, { 1, 95 }, { 1, 68 }, { 1, 42 }, { 1, 14 }, { 49, 0 }, { 1, 24 }, { 1, 248 }, { 10, 255 }, { 1, 233 }, { 1, 207 }, { 1, 180 }, { 1, 154 }, { 1, 128 }, { 1, 101 }, { 1, 75 }, { 1, 49 }, { 1, 23 }, { 1, 2 }, { 59, 0 }, { 1, 62 }, { 1, 187 }, { 1, 208 }, { 1, 187 }, { 1, 161 }, { 1, 135 }, { 1, 108 }, { 1, 82 }, { 1, 56 }, { 1, 29 }, { 1, 5 }, { 1804, 0 } }, 
	LogoRLEFrame{ { 1717, 0 }, { 1, 16 }, { 1, 160 }, { 1, 162 }, { 1, 39 }, { 76, 0 }, { 1, 154 }, { 2, 255 }, { 1, 226 }, { 1, 19 }, { 74, 0 }, { 1, 9 }, { 1, 242 }, { 3, 255 }, { 1, 134 }, { 74, 0 }, { 1, 86 }, { 4, 255 }, { 1, 242 }, { 1, 18 }, { 73, 0 }, { 1, 177 }, { 5, 255 }, { 1, 132 }, { 72, 0 }, { 1, 18 }, { 1, 249 }, { 5, 255 }, { 1, 241 }, { 1, 17 }, { 71, 0 }, { 1, 102 }, { 7, 255 }, { 1, 115 }, { 71, 0 }, { 1, 193 }, { 7, 255 }, { 1, 206 }, { 54, 0 }, { 1, 2 }, { 1, 12 }, { 14, 0 }, { 1, 29 }, { 9, 255 }, { 1, 43 }, { 53, 0 }, { 1, 31 }, { 1, 6 }, { 14, 0 }, { 1, 118 }, { 9, 255 }, { 1, 134 }, { 52, 0 }, { 1, 6 }, { 1, 52 }, { 15, 0 }, { 1, 209 }, { 9, 255 }, { 1, 222 }, { 52, 0 }, { 1, 34 }, { 1, 39 }, { 14, 0 }, { 1, 44 }, { 10, 255 }, { 1, 250 }, { 51, 0 }, { 1, 1 }, { 1, 57 }, { 1, 17 }, { 14, 0 }, { 1, 135 }, { 11, 255 }, { 1, 3 }, { 50, 0 }, { 1, 22 }, { 1, 56 }, { 14, 0 }, { 1, 1 }, { 1, 224 }, { 10, 255 }, { 1, 205 }, { 51, 0 }, { 1, 47 }, { 1, 34 }, { 14, 0 }, { 1, 60 }, { 11, 255 }, { 1, 113 }, { 50, 0 }, { 1, 4 }, { 1, 61 }, { 1, 9 }, { 14, 0 }, { 1, 151 }, { 10, 255 }, { 1, 215 }, { 1, 8 }, { 19, 0 }, { 1, 5 }, { 30, 0 }, { 1, 21 }, { 1, 37 }, { 14, 0 }, { 1, 5 }, { 1, 236 }, { 9, 255 }, { 1, 251 }, { 1, 52 }, { 18, 0 }, { 1, 9 }, { 1, 27 }, { 31, 0 }, { 1, 35 }, { 1, 5 }, { 14, 0 }, { 1, 76 }, { 10, 255 }, { 1, 127 }, { 18, 0 }, { 1, 24 }, { 1, 37 }, { 32, 0 }, { 1, 24 }, { 15, 0 }, { 1, 167 }, { 9, 255 }, { 1, 203 }, { 1, 4 }, { 16, 0 }, { 1, 4 }, { 1, 43 }, { 1, 40 }, { 32, 0 }, { 1, 4 }, { 1, 2 }, { 14, 0 }, { 1, 12 }, { 1, 245 }, { 8, 255 }, { 1, 246 }, { 1, 40 }, { 16, 0 }, { 1, 11 }, { 1, 55 }, { 1, 39 }, { 49, 0 }, { 1, 93 }, { 9, 255 }, { 1, 110 }, { 5, 0 }, { 1, 11 }, { 1, 140 }, { 1, 216 }, { 1, 227 }, { 1, 181 }, { 1, 69 }, { 5, 0 }, { 1, 20 }, { 1, 59 }, { 1, 31 }, { 50, 0 }, { 1, 183 }, { 8, 255 }, { 1, 189 }, { 1, 1 }, { 4, 0 }, { 1, 52 }, { 1, 218 }, { 5, 255 }, { 1, 96 }, { 3, 0 }, { 1, 31 }, { 1, 60 }, { 1, 22 }, { 50, 0 }, { 1, 22 }, { 1, 251 }, { 7, 255 }, { 1, 240 }, { 1, 30 }, { 3, 0 }, { 1, 1 }, { 1, 117 }, { 1, 250 }, { 6, 255 }, { 1, 246 }, { 1, 45 }, { 1, 2 }, { 1, 42 }, { 1, 58 }, { 1, 14 }, { 51, 0 }, { 1, 109 }, { 8, 255 }, { 1, 94 }, { 3, 0 }, { 1, 25 }, { 1, 187 }, { 9, 255 }, { 1, 217 }, { 1, 57 }, { 1, 51 }, { 1, 8 }, { 52, 0 }, { 1, 199 }, { 7, 255 }, { 1, 174 }, { 3, 0 }, { 1, 76 }, { 1, 235 }, { 11, 255 }, { 1, 180 }, { 53, 0 }, { 1, 35 }, { 7, 255 }, { 1, 233 }, { 1, 21 }, { 1, 0 }, { 1, 8 }, { 1, 147 }, { 14, 255 }, { 1, 107 }, { 52, 0 }, { 1, 125 }, { 7, 255 }, { 1, 99 }, { 1, 0 }, { 1, 43 }, { 1, 210 }, { 15, 255 }, { 1, 249 }, { 1, 53 }, { 51, 0 }, { 1, 215 }, { 7, 255 }, { 1, 159 }, { 1, 112 }, { 1, 246 }, { 17, 255 }, { 1, 224 }, { 1, 17 }, { 49, 0 }, { 1, 50 }, { 29, 255 }, { 1, 179 }, { 1, 1 }, { 48, 0 }, { 1, 141 }, { 30, 255 }, { 1, 119 }, { 47, 0 }, { 1, 2 }, { 1, 229 }, { 30, 255 }, { 1, 252 }, { 1, 45 }, { 46, 0 }, { 1, 67 }, { 32, 255 }, { 1, 139 }, { 46, 0 }, { 1, 157 }, { 32, 255 }, { 1, 108 }, { 45, 0 }, { 1, 7 }, { 1, 240 }, { 31, 255 }, { 1, 212 }, { 1, 21 }, { 45, 0 }, { 1, 58 }, { 29, 255 }, { 1, 244 }, { 1, 208 }, { 1, 127 }, { 1, 17 }, { 46, 0 }, { 1, 57 }, { 19, 255 }, { 1, 238 }, { 1, 213 }, { 1, 188 }, { 1, 163 }, { 1, 138 }, { 1, 113 }, { 1, 88 }, { 1, 63 }, { 1, 38 }, { 1, 13 }, { 51, 0 }, { 1, 191 }, { 7, 255 }, { 1, 253 }, { 1, 232 }, { 1, 207 }, { 1, 182 }, { 1, 157 }, { 1, 132 }, { 1, 107 }, { 1, 82 }, { 1, 57 }, { 1, 32 }, { 1, 8 }, { 61, 0 }, { 1, 6 }, { 1, 103 }, { 1, 139 }, { 1, 126 }, { 1, 101 }, { 1, 76 }, { 1, 51 }, { 1, 26 }, { 1, 3 }, { 1725, 0 } }, 
	LogoRLEFrame{ { 1718, 0 }, { 1, 88 }, { 1, 155 }, { 1, 71 }, { 76, 0 }, { 1, 47 }, { 1, 252 }, { 2, 255 }, { 1, 85 }, { 75, 0 }, { 1, 140 }, { 3, 255 }, { 1, 230 }, { 1, 9 }, { 74, 0 }, { 1, 225 }, { 4, 255 }, { 1, 115 }, { 57, 0 }, { 1, 14 }, { 15, 0 }, { 1, 55 }, { 5, 255 }, { 1, 234 }, { 1, 12 }, { 55, 0 }, { 1, 1 }, { 1, 35 }, { 15, 0 }, { 1, 141 }, { 6, 255 }, { 1, 121 }, { 55, 0 }, { 1, 28 }, { 1, 29 }, { 14, 0 }, { 1, 1 }, { 1, 226 }, { 6, 255 }, { 1, 236 }, { 1, 7 }, { 53, 0 }, { 1, 1 }, { 1, 57 }, { 1, 14 }, { 14, 0 }, { 1, 56 }, { 8, 255 }, { 1, 84 }, { 53, 0 }, { 1, 21 }, { 1, 54 }, { 15, 0 }, { 1, 142 }, { 8, 255 }, { 1, 181 }, { 53, 0 }, { 1, 47 }, { 1, 32 }, { 14, 0 }, { 1, 1 }, { 1, 227 }, { 8, 255 }, { 1, 252 }, { 1, 25 }, { 51, 0 }, { 1, 9 }, { 1, 62 }, { 1, 9 }, { 14, 0 }, { 1, 57 }, { 10, 255 }, { 1, 118 }, { 51, 0 }, { 1, 29 }, { 1, 46 }, { 15, 0 }, { 1, 143 }, { 10, 255 }, { 1, 170 }, { 51, 0 }, { 1, 46 }, { 1, 13 }, { 14, 0 }, { 1, 1 }, { 1, 227 }, { 10, 255 }, { 1, 185 }, { 21, 0 }, { 1, 4 }, { 1, 17 }, { 27, 0 }, { 1, 2 }, { 1, 40 }, { 15, 0 }, { 1, 58 }, { 11, 255 }, { 1, 167 }, { 20, 0 }, { 1, 16 }, { 1, 33 }, { 28, 0 }, { 1, 17 }, { 1, 8 }, { 15, 0 }, { 1, 144 }, { 11, 255 }, { 1, 84 }, { 18, 0 }, { 1, 1 }, { 1, 34 }, { 1, 40 }, { 29, 0 }, { 1, 9 }, { 15, 0 }, { 1, 1 }, { 1, 228 }, { 10, 255 }, { 1, 219 }, { 1, 8 }, { 17, 0 }, { 1, 8 }, { 1, 51 }, { 1, 40 }, { 46, 0 }, { 1, 59 }, { 10, 255 }, { 1, 253 }, { 1, 60 }, { 17, 0 }, { 1, 16 }, { 1, 58 }, { 1, 35 }, { 47, 0 }, { 1, 145 }, { 10, 255 }, { 1, 141 }, { 17, 0 }, { 1, 27 }, { 1, 61 }, { 1, 26 }, { 47, 0 }, { 1, 1 }, { 1, 229 }, { 9, 255 }, { 1, 217 }, { 1, 9 }, { 15, 0 }, { 1, 1 }, { 1, 38 }, { 1, 59 }, { 1, 18 }, { 48, 0 }, { 1, 60 }, { 9, 255 }, { 1, 253 }, { 1, 57 }, { 15, 0 }, { 1, 2 }, { 1, 47 }, { 1, 55 }, { 1, 11 }, { 49, 0 }, { 1, 146 }, { 9, 255 }, { 1, 137 }, { 6, 0 }, { 1, 42 }, { 1, 108 }, { 1, 118 }, { 1, 73 }, { 1, 3 }, { 4, 0 }, { 1, 2 }, { 1, 47 }, { 1, 44 }, { 1, 4 }, { 49, 0 }, { 1, 2 }, { 1, 229 }, { 8, 255 }, { 1, 214 }, { 1, 7 }, { 4, 0 }, { 1, 4 }, { 1, 129 }, { 1, 251 }, { 3, 255 }, { 1, 215 }, { 1, 35 }, { 2, 0 }, { 1, 3 }, { 1, 46 }, { 1, 27 }, { 51, 0 }, { 1, 61 }, { 8, 255 }, { 1, 252 }, { 1, 53 }, { 4, 0 }, { 1, 33 }, { 1, 198 }, { 6, 255 }, { 1, 211 }, { 1, 12 }, { 1, 3 }, { 1, 34 }, { 1, 10 }, { 52, 0 }, { 1, 147 }, { 8, 255 }, { 1, 133 }, { 4, 0 }, { 1, 90 }, { 1, 241 }, { 8, 255 }, { 1, 172 }, { 1, 10 }, { 53, 0 }, { 1, 2 }, { 1, 230 }, { 7, 255 }, { 1, 211 }, { 1, 6 }, { 2, 0 }, { 1, 13 }, { 1, 162 }, { 11, 255 }, { 1, 121 }, { 53, 0 }, { 1, 62 }, { 7, 255 }, { 1, 251 }, { 1, 50 }, { 2, 0 }, { 1, 54 }, { 1, 220 }, { 12, 255 }, { 1, 253 }, { 1, 73 }, { 52, 0 }, { 1, 148 }, { 7, 255 }, { 1, 133 }, { 1, 0 }, { 1, 2 }, { 1, 121 }, { 1, 251 }, { 14, 255 }, { 1, 239 }, { 1, 36 }, { 50, 0 }, { 1, 2 }, { 1, 231 }, { 7, 255 }, { 1, 100 }, { 1, 28 }, { 1, 191 }, { 17, 255 }, { 1, 212 }, { 1, 12 }, { 49, 0 }, { 1, 63 }, { 8, 255 }, { 1, 249 }, { 1, 245 }, { 19, 255 }, { 1, 172 }, { 1, 1 }, { 48, 0 }, { 1, 149 }, { 30, 255 }, { 1, 122 }, { 47, 0 }, { 1, 2 }, { 1, 232 }, { 30, 255 }, { 1, 253 }, { 1, 68 }, { 46, 0 }, { 1, 64 }, { 32, 255 }, { 1, 205 }, { 46, 0 }, { 1, 150 }, { 32, 255 }, { 1, 229 }, { 45, 0 }, { 1, 3 }, { 1, 232 }, { 32, 255 }, { 1, 150 }, { 45, 0 }, { 1, 58 }, { 31, 255 }, { 1, 248 }, { 1, 154 }, { 1, 5 }, { 45, 0 }, { 1, 77 }, { 23, 255 }, { 1, 249 }, { 1, 224 }, { 1, 199 }, { 1, 174 }, { 1, 149 }, { 1, 124 }, { 1, 99 }, { 1, 72 }, { 1, 13 }, { 47, 0 }, { 1, 18 }, { 1, 241 }, { 12, 255 }, { 1, 244 }, { 1, 219 }, { 1, 194 }, { 1, 169 }, { 1, 144 }, { 1, 118 }, { 1, 93 }, { 1, 68 }, { 1, 43 }, { 1, 18 }, { 57, 0 }, { 1, 63 }, { 1, 213 }, { 1, 250 }, { 1, 239 }, { 1, 214 }, { 1, 188 }, { 1, 163 }, { 1, 138 }, { 1, 113 }, { 1, 88 }, { 1, 63 }, { 1, 38 }, { 1, 13 }, { 490, 0 }, { 1, 1 }, { 1, 20 }, { 1, 4 }, { 75, 0 }, { 1, 1 }, { 1, 24 }, { 1, 45 }, { 1, 8 }, { 75, 0 }, { 1, 15 }, { 1, 53 }, { 1, 46 }, { 1, 4 }, { 74, 0 }, { 1, 4 }, { 1, 39 }, { 1, 60 }, { 1, 26 }, { 75, 0 }, { 1, 17 }, { 1, 57 }, { 1, 47 }, { 1, 8 }, { 75, 0 }, { 1, 28 }, { 1, 50 }, { 1, 21 }, { 75, 0 }, { 1, 1 }, { 1, 24 }, { 1, 17 }, { 77, 0 }, { 1, 2 }, { 678, 0 } }, 
	LogoRLEFrame{ { 1637, 0 }, { 1, 4 }, { 1, 151 }, { 1, 183 }, { 1, 74 }, { 61, 0 }, { 1, 12 }, { 14, 0 }, { 1, 111 }, { 2, 255 }, { 1, 251 }, { 1, 68 }, { 59, 0 }, { 1, 17 }, { 1, 18 }, { 14, 0 }, { 1, 199 }, { 3, 255 }, { 1, 212 }, { 1, 2 }, { 58, 0 }, { 1, 51 }, { 1, 4 }, { 13, 0 }, { 1, 27 }, { 5, 255 }, { 1, 92 }, { 57, 0 }, { 1, 21 }, { 1, 51 }, { 14, 0 }, { 1, 109 }, { 5, 255 }, { 1, 221 }, { 1, 5 }, { 56, 0 }, { 1, 47 }, { 1, 30 }, { 14, 0 }, { 1, 191 }, { 6, 255 }, { 1, 104 }, { 55, 0 }, { 1, 9 }, { 1, 62 }, { 1, 7 }, { 13, 0 }, { 1, 21 }, { 1, 252 }, { 6, 255 }, { 1, 224 }, { 1, 2 }, { 54, 0 }, { 1, 34 }, { 1, 46 }, { 14, 0 }, { 1, 100 }, { 8, 255 }, { 1, 70 }, { 54, 0 }, { 1, 55 }, { 1, 21 }, { 14, 0 }, { 1, 183 }, { 8, 255 }, { 1, 169 }, { 53, 0 }, { 1, 9 }, { 1, 50 }, { 14, 0 }, { 1, 16 }, { 1, 250 }, { 8, 255 }, { 1, 249 }, { 1, 19 }, { 52, 0 }, { 1, 26 }, { 1, 17 }, { 14, 0 }, { 1, 92 }, { 10, 255 }, { 1, 111 }, { 52, 0 }, { 1, 27 }, { 15, 0 }, { 1, 175 }, { 10, 255 }, { 1, 162 }, { 52, 0 }, { 1, 8 }, { 14, 0 }, { 1, 11 }, { 1, 246 }, { 10, 255 }, { 1, 180 }, { 67, 0 }, { 1, 84 }, { 11, 255 }, { 1, 161 }, { 67, 0 }, { 1, 167 }, { 11, 255 }, { 1, 79 }, { 66, 0 }, { 1, 7 }, { 1, 242 }, { 10, 255 }, { 1, 215 }, { 1, 7 }, { 66, 0 }, { 1, 76 }, { 10, 255 }, { 1, 253 }, { 1, 57 }, { 67, 0 }, { 1, 158 }, { 10, 255 }, { 1, 141 }, { 67, 0 }, { 1, 4 }, { 1, 237 }, { 9, 255 }, { 1, 219 }, { 1, 9 }, { 67, 0 }, { 1, 68 }, { 10, 255 }, { 1, 61 }, { 68, 0 }, { 1, 150 }, { 9, 255 }, { 1, 147 }, { 6, 0 }, { 1, 36 }, { 1, 110 }, { 1, 128 }, { 1, 88 }, { 1, 10 }, { 57, 0 }, { 1, 2 }, { 1, 231 }, { 8, 255 }, { 1, 222 }, { 1, 11 }, { 4, 0 }, { 1, 1 }, { 1, 112 }, { 1, 247 }, { 3, 255 }, { 1, 233 }, { 1, 64 }, { 56, 0 }, { 1, 59 }, { 9, 255 }, { 1, 66 }, { 4, 0 }, { 1, 22 }, { 1, 183 }, { 6, 255 }, { 1, 237 }, { 1, 34 }, { 55, 0 }, { 1, 142 }, { 8, 255 }, { 1, 152 }, { 4, 0 }, { 1, 72 }, { 1, 232 }, { 8, 255 }, { 1, 212 }, { 1, 13 }, { 54, 0 }, { 1, 224 }, { 7, 255 }, { 1, 226 }, { 1, 14 }, { 2, 0 }, { 1, 6 }, { 1, 142 }, { 11, 255 }, { 1, 177 }, { 1, 1 }, { 52, 0 }, { 1, 51 }, { 8, 255 }, { 1, 72 }, { 2, 0 }, { 1, 40 }, { 1, 207 }, { 13, 255 }, { 1, 132 }, { 52, 0 }, { 1, 134 }, { 7, 255 }, { 1, 162 }, { 2, 0 }, { 1, 101 }, { 1, 245 }, { 15, 255 }, { 1, 88 }, { 51, 0 }, { 1, 217 }, { 7, 255 }, { 1, 137 }, { 1, 17 }, { 1, 172 }, { 17, 255 }, { 1, 246 }, { 1, 51 }, { 49, 0 }, { 1, 43 }, { 9, 255 }, { 1, 242 }, { 19, 255 }, { 1, 227 }, { 1, 24 }, { 48, 0 }, { 1, 126 }, { 30, 255 }, { 1, 198 }, { 1, 7 }, { 47, 0 }, { 1, 208 }, { 31, 255 }, { 1, 151 }, { 46, 0 }, { 1, 35 }, { 33, 255 }, { 1, 34 }, { 45, 0 }, { 1, 118 }, { 33, 255 }, { 1, 50 }, { 45, 0 }, { 1, 200 }, { 32, 255 }, { 1, 219 }, { 1, 2 }, { 44, 0 }, { 1, 22 }, { 31, 255 }, { 1, 252 }, { 1, 184 }, { 1, 25 }, { 45, 0 }, { 1, 39 }, { 23, 255 }, { 1, 247 }, { 1, 221 }, { 1, 195 }, { 1, 169 }, { 1, 143 }, { 1, 117 }, { 1, 91 }, { 1, 66 }, { 1, 20 }, { 47, 0 }, { 1, 4 }, { 1, 219 }, { 12, 255 }, { 1, 250 }, { 1, 225 }, { 1, 199 }, { 1, 173 }, { 1, 147 }, { 1, 121 }, { 1, 95 }, { 1, 69 }, { 1, 43 }, { 1, 17 }, { 57, 0 }, { 1, 45 }, { 1, 206 }, { 1, 253 }, { 1, 252 }, { 1, 228 }, { 1, 202 }, { 1, 176 }, { 1, 150 }, { 1, 125 }, { 1, 99 }, { 1, 73 }, { 1, 47 }, { 1, 21 }, { 1, 1 }, { 68, 0 }, { 1, 5 }, { 1, 2 }, { 343, 0 }, { 1, 6 }, { 77, 0 }, { 1, 13 }, { 1, 35 }, { 1, 7 }, { 75, 0 }, { 1, 10 }, { 1, 44 }, { 1, 46 }, { 1, 3 }, { 74, 0 }, { 1, 1 }, { 1, 31 }, { 1, 60 }, { 1, 32 }, { 75, 0 }, { 1, 15 }, { 1, 53 }, { 1, 52 }, { 1, 12 }, { 75, 0 }, { 1, 28 }, { 1, 58 }, { 1, 31 }, { 1, 1 }, { 74, 0 }, { 1, 1 }, { 1, 34 }, { 1, 30 }, { 1, 3 }, { 76, 0 }, { 1, 13 }, { 1, 2 }, { 834, 0 } }, 
	LogoRLEFrame{ { 1383, 0 }, { 1, 8 }, { 1, 2 }, { 78, 0 }, { 1, 35 }, { 78, 0 }, { 1, 15 }, { 1, 41 }, { 13, 0 }, { 1, 16 }, { 1, 183 }, { 1, 198 }, { 1, 75 }, { 61, 0 }, { 1, 46 }, { 1, 26 }, { 13, 0 }, { 1, 142 }, { 2, 255 }, { 1, 250 }, { 1, 59 }, { 59, 0 }, { 1, 9 }, { 1, 61 }, { 1, 5 }, { 13, 0 }, { 1, 226 }, { 3, 255 }, { 1, 200 }, { 59, 0 }, { 1, 34 }, { 1, 44 }, { 13, 0 }, { 1, 53 }, { 5, 255 }, { 1, 80 }, { 57, 0 }, { 1, 1 }, { 1, 58 }, { 1, 21 }, { 13, 0 }, { 1, 135 }, { 5, 255 }, { 1, 213 }, { 1, 3 }, { 56, 0 }, { 1, 17 }, { 1, 57 }, { 1, 1 }, { 13, 0 }, { 1, 216 }, { 6, 255 }, { 1, 95 }, { 56, 0 }, { 1, 35 }, { 1, 27 }, { 13, 0 }, { 1, 42 }, { 7, 255 }, { 1, 216 }, { 56, 0 }, { 1, 43 }, { 1, 1 }, { 13, 0 }, { 1, 124 }, { 8, 255 }, { 1, 60 }, { 54, 0 }, { 1, 6 }, { 1, 21 }, { 14, 0 }, { 1, 205 }, { 8, 255 }, { 1, 161 }, { 54, 0 }, { 1, 10 }, { 14, 0 }, { 1, 32 }, { 9, 255 }, { 1, 246 }, { 1, 16 }, { 68, 0 }, { 1, 113 }, { 10, 255 }, { 1, 105 }, { 68, 0 }, { 1, 194 }, { 10, 255 }, { 1, 147 }, { 67, 0 }, { 1, 23 }, { 1, 253 }, { 10, 255 }, { 1, 165 }, { 67, 0 }, { 1, 102 }, { 11, 255 }, { 1, 135 }, { 67, 0 }, { 1, 183 }, { 11, 255 }, { 1, 54 }, { 66, 0 }, { 1, 15 }, { 1, 250 }, { 10, 255 }, { 1, 188 }, { 67, 0 }, { 1, 91 }, { 10, 255 }, { 1, 244 }, { 1, 33 }, { 67, 0 }, { 1, 173 }, { 10, 255 }, { 1, 109 }, { 67, 0 }, { 1, 9 }, { 1, 245 }, { 9, 255 }, { 1, 196 }, { 1, 2 }, { 67, 0 }, { 1, 80 }, { 9, 255 }, { 1, 247 }, { 1, 39 }, { 68, 0 }, { 1, 162 }, { 9, 255 }, { 1, 119 }, { 6, 0 }, { 1, 68 }, { 1, 147 }, { 1, 163 }, { 1, 123 }, { 1, 28 }, { 57, 0 }, { 1, 5 }, { 1, 238 }, { 8, 255 }, { 1, 204 }, { 1, 3 }, { 4, 0 }, { 1, 9 }, { 1, 152 }, { 4, 255 }, { 1, 248 }, { 1, 83 }, { 56, 0 }, { 1, 69 }, { 8, 255 }, { 1, 250 }, { 1, 46 }, { 4, 0 }, { 1, 45 }, { 1, 213 }, { 6, 255 }, { 1, 245 }, { 1, 49 }, { 55, 0 }, { 1, 151 }, { 8, 255 }, { 1, 129 }, { 4, 0 }, { 1, 105 }, { 1, 247 }, { 8, 255 }, { 1, 226 }, { 1, 23 }, { 53, 0 }, { 1, 1 }, { 1, 230 }, { 7, 255 }, { 1, 211 }, { 1, 6 }, { 2, 0 }, { 1, 18 }, { 1, 175 }, { 11, 255 }, { 1, 198 }, { 1, 7 }, { 52, 0 }, { 1, 58 }, { 7, 255 }, { 1, 252 }, { 1, 53 }, { 2, 0 }, { 1, 62 }, { 1, 226 }, { 13, 255 }, { 1, 160 }, { 52, 0 }, { 1, 140 }, { 7, 255 }, { 1, 146 }, { 1, 0 }, { 1, 3 }, { 1, 128 }, { 1, 252 }, { 15, 255 }, { 1, 116 }, { 51, 0 }, { 1, 221 }, { 7, 255 }, { 1, 144 }, { 1, 29 }, { 1, 194 }, { 17, 255 }, { 1, 253 }, { 1, 75 }, { 49, 0 }, { 1, 47 }, { 9, 255 }, { 1, 252 }, { 19, 255 }, { 1, 241 }, { 1, 42 }, { 48, 0 }, { 1, 129 }, { 30, 255 }, { 1, 220 }, { 1, 18 }, { 47, 0 }, { 1, 211 }, { 31, 255 }, { 1, 180 }, { 46, 0 }, { 1, 37 }, { 33, 255 }, { 1, 44 }, { 45, 0 }, { 1, 118 }, { 33, 255 }, { 1, 36 }, { 45, 0 }, { 1, 200 }, { 32, 255 }, { 1, 188 }, { 45, 0 }, { 1, 20 }, { 31, 255 }, { 1, 227 }, { 1, 138 }, { 1, 8 }, { 45, 0 }, { 1, 34 }, { 22, 255 }, { 1, 239 }, { 1, 212 }, { 1, 185 }, { 1, 158 }, { 1, 131 }, { 1, 104 }, { 1, 77 }, { 1, 50 }, { 1, 23 }, { 1, 1 }, { 47, 0 }, { 1, 2 }, { 1, 211 }, { 11, 255 }, { 1, 250 }, { 1, 225 }, { 1, 198 }, { 1, 171 }, { 1, 144 }, { 1, 117 }, { 1, 91 }, { 1, 64 }, { 1, 37 }, { 1, 10 }, { 58, 0 }, { 1, 36 }, { 1, 190 }, { 1, 243 }, { 1, 239 }, { 1, 212 }, { 1, 185 }, { 1, 158 }, { 1, 131 }, { 1, 104 }, { 1, 77 }, { 1, 50 }, { 1, 23 }, { 1, 1 }, { 415, 0 }, { 1, 5 }, { 1, 23 }, { 1, 2 }, { 75, 0 }, { 1, 4 }, { 1, 33 }, { 1, 42 }, { 1, 3 }, { 75, 0 }, { 1, 25 }, { 1, 58 }, { 1, 35 }, { 75, 0 }, { 1, 10 }, { 1, 48 }, { 1, 55 }, { 1, 16 }, { 75, 0 }, { 1, 27 }, { 1, 61 }, { 1, 38 }, { 1, 3 }, { 74, 0 }, { 1, 1 }, { 1, 38 }, { 1, 42 }, { 1, 12 }, { 75, 0 }, { 1, 4 }, { 1, 25 }, { 1, 9 }, { 77, 0 }, { 1, 1 }, { 913, 0 } }, 
	LogoRLEFrame{ { 1478, 0 }, { 1, 38 }, { 1, 100 }, { 1, 33 }, { 76, 0 }, { 1, 17 }, { 1, 233 }, { 1, 255 }, { 1, 246 }, { 1, 62 }, { 75, 0 }, { 1, 102 }, { 3, 255 }, { 1, 223 }, { 1, 6 }, { 74, 0 }, { 1, 184 }, { 4, 255 }, { 1, 107 }, { 73, 0 }, { 1, 16 }, { 1, 250 }, { 4, 255 }, { 1, 231 }, { 1, 10 }, { 72, 0 }, { 1, 94 }, { 6, 255 }, { 1, 121 }, { 72, 0 }, { 1, 176 }, { 6, 255 }, { 1, 239 }, { 1, 11 }, { 70, 0 }, { 1, 12 }, { 1, 247 }, { 7, 255 }, { 1, 96 }, { 70, 0 }, { 1, 86 }, { 8, 255 }, { 1, 196 }, { 70, 0 }, { 1, 169 }, { 9, 255 }, { 1, 40 }, { 68, 0 }, { 1, 8 }, { 1, 243 }, { 9, 255 }, { 1, 138 }, { 68, 0 }, { 1, 78 }, { 10, 255 }, { 1, 206 }, { 68, 0 }, { 1, 161 }, { 10, 255 }, { 1, 223 }, { 67, 0 }, { 1, 5 }, { 1, 238 }, { 10, 255 }, { 1, 218 }, { 67, 0 }, { 1, 70 }, { 11, 255 }, { 1, 142 }, { 67, 0 }, { 1, 153 }, { 10, 255 }, { 1, 251 }, { 1, 46 }, { 66, 0 }, { 1, 2 }, { 1, 233 }, { 10, 255 }, { 1, 134 }, { 67, 0 }, { 1, 62 }, { 10, 255 }, { 1, 215 }, { 1, 7 }, { 67, 0 }, { 1, 145 }, { 9, 255 }, { 1, 253 }, { 1, 57 }, { 67, 0 }, { 1, 1 }, { 1, 226 }, { 9, 255 }, { 1, 142 }, { 68, 0 }, { 1, 54 }, { 9, 255 }, { 1, 221 }, { 1, 10 }, { 5, 0 }, { 1, 10 }, { 1, 76 }, { 1, 103 }, { 1, 70 }, { 1, 7 }, { 58, 0 }, { 1, 137 }, { 9, 255 }, { 1, 65 }, { 5, 0 }, { 1, 55 }, { 1, 218 }, { 3, 255 }, { 1, 228 }, { 1, 70 }, { 57, 0 }, { 1, 219 }, { 8, 255 }, { 1, 151 }, { 4, 0 }, { 1, 1 }, { 1, 118 }, { 1, 251 }, { 5, 255 }, { 1, 241 }, { 1, 41 }, { 55, 0 }, { 1, 46 }, { 8, 255 }, { 1, 226 }, { 1, 13 }, { 3, 0 }, { 1, 22 }, { 1, 184 }, { 8, 255 }, { 1, 219 }, { 1, 17 }, { 54, 0 }, { 1, 129 }, { 8, 255 }, { 1, 72 }, { 3, 0 }, { 1, 68 }, { 1, 231 }, { 10, 255 }, { 1, 186 }, { 1, 3 }, { 46, 0 }, { 1, 5 }, { 6, 0 }, { 1, 211 }, { 7, 255 }, { 1, 160 }, { 2, 0 }, { 1, 4 }, { 1, 133 }, { 1, 253 }, { 12, 255 }, { 1, 142 }, { 45, 0 }, { 1, 1 }, { 1, 7 }, { 5, 0 }, { 1, 38 }, { 7, 255 }, { 1, 231 }, { 1, 17 }, { 1, 0 }, { 1, 30 }, { 1, 197 }, { 15, 255 }, { 1, 96 }, { 44, 0 }, { 1, 12 }, { 1, 1 }, { 5, 0 }, { 1, 121 }, { 7, 255 }, { 1, 165 }, { 1, 0 }, { 1, 82 }, { 1, 238 }, { 16, 255 }, { 1, 248 }, { 1, 57 }, { 42, 0 }, { 1, 4 }, { 1, 14 }, { 6, 0 }, { 1, 204 }, { 7, 255 }, { 1, 250 }, { 1, 187 }, { 19, 255 }, { 1, 231 }, { 1, 28 }, { 41, 0 }, { 1, 16 }, { 1, 7 }, { 5, 0 }, { 1, 31 }, { 30, 255 }, { 1, 203 }, { 1, 9 }, { 39, 0 }, { 1, 6 }, { 1, 18 }, { 6, 0 }, { 1, 113 }, { 31, 255 }, { 1, 159 }, { 39, 0 }, { 1, 17 }, { 1, 9 }, { 6, 0 }, { 1, 196 }, { 32, 255 }, { 1, 44 }, { 37, 0 }, { 1, 7 }, { 1, 19 }, { 6, 0 }, { 1, 24 }, { 1, 253 }, { 32, 255 }, { 1, 66 }, { 37, 0 }, { 1, 17 }, { 1, 10 }, { 6, 0 }, { 1, 105 }, { 32, 255 }, { 1, 232 }, { 1, 9 }, { 36, 0 }, { 1, 7 }, { 1, 19 }, { 1, 1 }, { 6, 0 }, { 1, 187 }, { 31, 255 }, { 1, 203 }, { 1, 37 }, { 37, 0 }, { 1, 16 }, { 1, 10 }, { 7, 0 }, { 1, 226 }, { 23, 255 }, { 1, 251 }, { 1, 224 }, { 1, 196 }, { 1, 168 }, { 1, 140 }, { 1, 112 }, { 1, 84 }, { 1, 36 }, { 38, 0 }, { 1, 4 }, { 1, 18 }, { 8, 0 }, { 1, 194 }, { 14, 255 }, { 1, 248 }, { 1, 220 }, { 1, 192 }, { 1, 164 }, { 1, 136 }, { 1, 108 }, { 1, 80 }, { 1, 52 }, { 1, 24 }, { 1, 2 }, { 45, 0 }, { 1, 12 }, { 1, 5 }, { 8, 0 }, { 1, 50 }, { 1, 230 }, { 4, 255 }, { 1, 245 }, { 1, 216 }, { 1, 188 }, { 1, 160 }, { 1, 132 }, { 1, 104 }, { 1, 76 }, { 1, 48 }, { 1, 20 }, { 55, 0 }, { 1, 12 }, { 10, 0 }, { 1, 15 }, { 1, 64 }, { 1, 72 }, { 1, 44 }, { 1, 16 }, { 63, 0 }, { 1, 6 }, { 1, 2 }, { 78, 0 }, { 1, 3 }, { 122, 0 }, { 1, 8 }, { 77, 0 }, { 1, 22 }, { 1, 35 }, { 1, 3 }, { 75, 0 }, { 1, 17 }, { 1, 52 }, { 1, 36 }, { 75, 0 }, { 1, 5 }, { 1, 41 }, { 1, 58 }, { 1, 22 }, { 75, 0 }, { 1, 24 }, { 1, 58 }, { 1, 43 }, { 1, 6 }, { 74, 0 }, { 1, 1 }, { 1, 38 }, { 1, 53 }, { 1, 22 }, { 75, 0 }, { 1, 5 }, { 1, 36 }, { 1, 21 }, { 77, 0 }, { 1, 11 }, { 1070, 0 } }, 
	LogoRLEFrame{ { 1479, 0 }, { 1, 41 }, { 1, 68 }, { 1, 5 }, { 76, 0 }, { 1, 52 }, { 1, 251 }, { 1, 255 }, { 1, 199 }, { 1, 11 }, { 75, 0 }, { 1, 164 }, { 3, 255 }, { 1, 137 }, { 74, 0 }, { 1, 8 }, { 1, 242 }, { 3, 255 }, { 1, 245 }, { 1, 24 }, { 73, 0 }, { 1, 80 }, { 5, 255 }, { 1, 145 }, { 73, 0 }, { 1, 166 }, { 5, 255 }, { 1, 248 }, { 1, 29 }, { 71, 0 }, { 1, 8 }, { 1, 243 }, { 6, 255 }, { 1, 147 }, { 71, 0 }, { 1, 81 }, { 7, 255 }, { 1, 239 }, { 1, 8 }, { 70, 0 }, { 1, 167 }, { 8, 255 }, { 1, 87 }, { 69, 0 }, { 1, 9 }, { 1, 244 }, { 8, 255 }, { 1, 184 }, { 69, 0 }, { 1, 83 }, { 9, 255 }, { 1, 253 }, { 1, 28 }, { 68, 0 }, { 1, 169 }, { 10, 255 }, { 1, 82 }, { 67, 0 }, { 1, 10 }, { 1, 244 }, { 10, 255 }, { 1, 95 }, { 67, 0 }, { 1, 84 }, { 11, 255 }, { 1, 77 }, { 67, 0 }, { 1, 170 }, { 10, 255 }, { 1, 240 }, { 1, 7 }, { 66, 0 }, { 1, 11 }, { 1, 245 }, { 10, 255 }, { 1, 133 }, { 67, 0 }, { 1, 86 }, { 10, 255 }, { 1, 212 }, { 1, 7 }, { 67, 0 }, { 1, 172 }, { 9, 255 }, { 1, 252 }, { 1, 53 }, { 67, 0 }, { 1, 12 }, { 1, 246 }, { 9, 255 }, { 1, 135 }, { 62, 0 }, { 1, 2 }, { 5, 0 }, { 1, 88 }, { 9, 255 }, { 1, 214 }, { 1, 7 }, { 61, 0 }, { 1, 5 }, { 1, 9 }, { 5, 0 }, { 1, 173 }, { 8, 255 }, { 1, 252 }, { 1, 55 }, { 5, 0 }, { 1, 11 }, { 1, 117 }, { 1, 162 }, { 1, 144 }, { 1, 77 }, { 52, 0 }, { 1, 29 }, { 5, 0 }, { 1, 12 }, { 1, 247 }, { 8, 255 }, { 1, 137 }, { 5, 0 }, { 1, 54 }, { 1, 221 }, { 4, 255 }, { 1, 182 }, { 1, 3 }, { 49, 0 }, { 1, 13 }, { 1, 29 }, { 5, 0 }, { 1, 89 }, { 8, 255 }, { 1, 215 }, { 1, 8 }, { 4, 0 }, { 1, 113 }, { 1, 250 }, { 6, 255 }, { 1, 137 }, { 49, 0 }, { 1, 49 }, { 1, 7 }, { 5, 0 }, { 1, 175 }, { 7, 255 }, { 1, 253 }, { 1, 56 }, { 3, 0 }, { 1, 18 }, { 1, 178 }, { 9, 255 }, { 1, 84 }, { 47, 0 }, { 1, 24 }, { 1, 46 }, { 5, 0 }, { 1, 13 }, { 1, 247 }, { 7, 255 }, { 1, 139 }, { 3, 0 }, { 1, 58 }, { 1, 225 }, { 10, 255 }, { 1, 243 }, { 1, 42 }, { 45, 0 }, { 1, 1 }, { 1, 56 }, { 1, 19 }, { 5, 0 }, { 1, 91 }, { 7, 255 }, { 1, 216 }, { 1, 8 }, { 1, 0 }, { 1, 1 }, { 1, 118 }, { 1, 251 }, { 12, 255 }, { 1, 217 }, { 1, 14 }, { 44, 0 }, { 1, 26 }, { 1, 52 }, { 6, 0 }, { 1, 177 }, { 6, 255 }, { 1, 253 }, { 1, 57 }, { 1, 0 }, { 1, 20 }, { 1, 182 }, { 15, 255 }, { 1, 176 }, { 1, 1 }, { 42, 0 }, { 1, 1 }, { 1, 56 }, { 1, 23 }, { 5, 0 }, { 1, 14 }, { 1, 248 }, { 6, 255 }, { 1, 243 }, { 1, 8 }, { 1, 62 }, { 1, 228 }, { 17, 255 }, { 1, 123 }, { 42, 0 }, { 1, 27 }, { 1, 55 }, { 6, 0 }, { 1, 92 }, { 8, 255 }, { 1, 222 }, { 1, 252 }, { 18, 255 }, { 1, 253 }, { 1, 72 }, { 40, 0 }, { 1, 2 }, { 1, 56 }, { 1, 26 }, { 6, 0 }, { 1, 178 }, { 29, 255 }, { 1, 238 }, { 1, 34 }, { 39, 0 }, { 1, 25 }, { 1, 55 }, { 1, 1 }, { 5, 0 }, { 1, 15 }, { 1, 249 }, { 30, 255 }, { 1, 203 }, { 1, 2 }, { 38, 0 }, { 1, 49 }, { 1, 19 }, { 6, 0 }, { 1, 94 }, { 32, 255 }, { 1, 64 }, { 37, 0 }, { 1, 11 }, { 1, 43 }, { 7, 0 }, { 1, 180 }, { 32, 255 }, { 1, 49 }, { 37, 0 }, { 1, 33 }, { 1, 7 }, { 6, 0 }, { 1, 16 }, { 1, 249 }, { 31, 255 }, { 1, 193 }, { 1, 1 }, { 36, 0 }, { 1, 1 }, { 1, 24 }, { 7, 0 }, { 1, 94 }, { 30, 255 }, { 1, 226 }, { 1, 137 }, { 1, 9 }, { 37, 0 }, { 1, 10 }, { 8, 0 }, { 1, 129 }, { 21, 255 }, { 1, 251 }, { 1, 225 }, { 1, 196 }, { 1, 167 }, { 1, 138 }, { 1, 109 }, { 1, 80 }, { 1, 51 }, { 1, 22 }, { 49, 0 }, { 1, 87 }, { 12, 255 }, { 1, 253 }, { 1, 230 }, { 1, 201 }, { 1, 172 }, { 1, 143 }, { 1, 114 }, { 1, 85 }, { 1, 56 }, { 1, 27 }, { 1, 2 }, { 57, 0 }, { 1, 1 }, { 1, 151 }, { 3, 255 }, { 1, 235 }, { 1, 206 }, { 1, 177 }, { 1, 148 }, { 1, 119 }, { 1, 90 }, { 1, 61 }, { 1, 32 }, { 1, 5 }, { 68, 0 }, { 1, 14 }, { 1, 32 }, { 1, 9 }, { 188, 0 }, { 1, 11 }, { 1, 23 }, { 1, 1 }, { 75, 0 }, { 1, 10 }, { 1, 42 }, { 1, 35 }, { 75, 0 }, { 1, 2 }, { 1, 34 }, { 1, 59 }, { 1, 25 }, { 75, 0 }, { 1, 18 }, { 1, 55 }, { 1, 48 }, { 1, 9 }, { 74, 0 }, { 1, 1 }, { 1, 37 }, { 1, 60 }, { 1, 28 }, { 75, 0 }, { 1, 5 }, { 1, 44 }, { 1, 33 }, { 1, 5 }, { 75, 0 }, { 1, 7 }, { 1, 22 }, { 1, 3 }, { 1226, 0 } }, 
	LogoRLEFrame{ { 1479, 0 }, { 1, 67 }, { 1, 135 }, { 1, 53 }, { 76, 0 }, { 1, 37 }, { 1, 247 }, { 1, 255 }, { 1, 248 }, { 1, 59 }, { 75, 0 }, { 1, 135 }, { 3, 255 }, { 1, 204 }, { 74, 0 }, { 1, 1 }, { 1, 224 }, { 4, 255 }, { 1, 75 }, { 73, 0 }, { 1, 61 }, { 5, 255 }, { 1, 203 }, { 73, 0 }, { 1, 152 }, { 6, 255 }, { 1, 74 }, { 71, 0 }, { 1, 6 }, { 1, 237 }, { 6, 255 }, { 1, 184 }, { 71, 0 }, { 1, 79 }, { 7, 255 }, { 1, 252 }, { 1, 23 }, { 70, 0 }, { 1, 170 }, { 8, 255 }, { 1, 110 }, { 69, 0 }, { 1, 14 }, { 1, 247 }, { 8, 255 }, { 1, 202 }, { 69, 0 }, { 1, 96 }, { 10, 255 }, { 1, 27 }, { 68, 0 }, { 1, 187 }, { 10, 255 }, { 1, 41 }, { 67, 0 }, { 1, 25 }, { 1, 253 }, { 10, 255 }, { 1, 40 }, { 67, 0 }, { 1, 114 }, { 10, 255 }, { 1, 221 }, { 62, 0 }, { 1, 2 }, { 5, 0 }, { 1, 205 }, { 10, 255 }, { 1, 121 }, { 62, 0 }, { 1, 15 }, { 4, 0 }, { 1, 40 }, { 10, 255 }, { 1, 210 }, { 1, 6 }, { 61, 0 }, { 1, 13 }, { 1, 15 }, { 4, 0 }, { 1, 131 }, { 9, 255 }, { 1, 250 }, { 1, 48 }, { 62, 0 }, { 1, 42 }, { 5, 0 }, { 1, 221 }, { 9, 255 }, { 1, 125 }, { 62, 0 }, { 1, 25 }, { 1, 32 }, { 4, 0 }, { 1, 57 }, { 9, 255 }, { 1, 203 }, { 1, 4 }, { 61, 0 }, { 1, 4 }, { 1, 57 }, { 1, 9 }, { 4, 0 }, { 1, 148 }, { 8, 255 }, { 1, 248 }, { 1, 42 }, { 5, 0 }, { 1, 18 }, { 1, 85 }, { 1, 104 }, { 1, 63 }, { 1, 1 }, { 52, 0 }, { 1, 32 }, { 1, 45 }, { 4, 0 }, { 1, 4 }, { 1, 235 }, { 8, 255 }, { 1, 116 }, { 5, 0 }, { 1, 76 }, { 1, 232 }, { 3, 255 }, { 1, 208 }, { 1, 27 }, { 50, 0 }, { 1, 4 }, { 1, 59 }, { 1, 15 }, { 4, 0 }, { 1, 75 }, { 8, 255 }, { 1, 196 }, { 1, 2 }, { 3, 0 }, { 1, 4 }, { 1, 137 }, { 6, 255 }, { 1, 194 }, { 1, 3 }, { 49, 0 }, { 1, 32 }, { 1, 48 }, { 5, 0 }, { 1, 166 }, { 7, 255 }, { 1, 245 }, { 1, 36 }, { 3, 0 }, { 1, 26 }, { 1, 194 }, { 8, 255 }, { 1, 131 }, { 48, 0 }, { 1, 4 }, { 1, 59 }, { 1, 18 }, { 4, 0 }, { 1, 12 }, { 1, 245 }, { 7, 255 }, { 1, 107 }, { 3, 0 }, { 1, 69 }, { 1, 234 }, { 9, 255 }, { 1, 253 }, { 1, 68 }, { 47, 0 }, { 1, 33 }, { 1, 51 }, { 5, 0 }, { 1, 92 }, { 7, 255 }, { 1, 188 }, { 1, 1 }, { 1, 0 }, { 1, 2 }, { 1, 128 }, { 1, 253 }, { 11, 255 }, { 1, 232 }, { 1, 24 }, { 45, 0 }, { 1, 2 }, { 1, 59 }, { 1, 20 }, { 5, 0 }, { 1, 183 }, { 6, 255 }, { 1, 241 }, { 1, 30 }, { 1, 0 }, { 1, 22 }, { 1, 187 }, { 14, 255 }, { 1, 188 }, { 1, 2 }, { 44, 0 }, { 1, 24 }, { 1, 44 }, { 5, 0 }, { 1, 22 }, { 1, 252 }, { 6, 255 }, { 1, 130 }, { 1, 0 }, { 1, 61 }, { 1, 229 }, { 16, 255 }, { 1, 124 }, { 44, 0 }, { 1, 46 }, { 1, 8 }, { 5, 0 }, { 1, 110 }, { 7, 255 }, { 1, 207 }, { 1, 134 }, { 1, 252 }, { 17, 255 }, { 1, 252 }, { 1, 63 }, { 42, 0 }, { 1, 10 }, { 1, 30 }, { 6, 0 }, { 1, 201 }, { 28, 255 }, { 1, 229 }, { 1, 21 }, { 41, 0 }, { 1, 24 }, { 1, 1 }, { 5, 0 }, { 1, 36 }, { 30, 255 }, { 1, 180 }, { 40, 0 }, { 1, 1 }, { 1, 10 }, { 6, 0 }, { 1, 127 }, { 31, 255 }, { 1, 56 }, { 47, 0 }, { 1, 218 }, { 31, 255 }, { 1, 69 }, { 46, 0 }, { 1, 54 }, { 31, 255 }, { 1, 226 }, { 1, 7 }, { 46, 0 }, { 1, 145 }, { 30, 255 }, { 1, 190 }, { 1, 29 }, { 47, 0 }, { 1, 216 }, { 23, 255 }, { 1, 232 }, { 1, 202 }, { 1, 171 }, { 1, 140 }, { 1, 109 }, { 1, 79 }, { 1, 28 }, { 49, 0 }, { 1, 220 }, { 14, 255 }, { 1, 251 }, { 1, 223 }, { 1, 192 }, { 1, 161 }, { 1, 130 }, { 1, 99 }, { 1, 69 }, { 1, 38 }, { 1, 8 }, { 56, 0 }, { 1, 104 }, { 6, 255 }, { 1, 244 }, { 1, 213 }, { 1, 182 }, { 1, 151 }, { 1, 120 }, { 1, 90 }, { 1, 59 }, { 1, 28 }, { 1, 2 }, { 65, 0 }, { 1, 61 }, { 1, 116 }, { 1, 111 }, { 1, 80 }, { 1, 49 }, { 1, 18 }, { 109, 0 }, { 1, 3 }, { 1, 9 }, { 76, 0 }, { 1, 3 }, { 2, 31 }, { 76, 0 }, { 1, 26 }, { 1, 57 }, { 1, 26 }, { 75, 0 }, { 1, 12 }, { 1, 49 }, { 1, 53 }, { 1, 14 }, { 74, 0 }, { 1, 1 }, { 1, 34 }, { 1, 61 }, { 1, 34 }, { 1, 1 }, { 74, 0 }, { 1, 5 }, { 1, 47 }, { 1, 46 }, { 1, 14 }, { 75, 0 }, { 1, 11 }, { 1, 35 }, { 1, 13 }, { 76, 0 }, { 1, 1 }, { 1, 7 }, { 1305, 0 } }, 
	LogoRLEFrame{ { 1400, 0 }, { 1, 1 }, { 77, 0 }, { 1, 1 }, { 1, 167 }, { 1, 246 }, { 1, 150 }, { 1, 1 }, { 75, 0 }, { 1, 82 }, { 3, 255 }, { 1, 102 }, { 75, 0 }, { 1, 182 }, { 3, 255 }, { 1, 222 }, { 1, 3 }, { 73, 0 }, { 1, 27 }, { 1, 252 }, { 4, 255 }, { 1, 89 }, { 73, 0 }, { 1, 123 }, { 5, 255 }, { 1, 209 }, { 72, 0 }, { 1, 1 }, { 1, 221 }, { 6, 255 }, { 1, 65 }, { 71, 0 }, { 1, 65 }, { 7, 255 }, { 1, 151 }, { 71, 0 }, { 1, 164 }, { 7, 255 }, { 1, 233 }, { 1, 3 }, { 69, 0 }, { 1, 16 }, { 1, 247 }, { 8, 255 }, { 1, 65 }, { 65, 0 }, { 1, 2 }, { 3, 0 }, { 1, 106 }, { 9, 255 }, { 1, 142 }, { 64, 0 }, { 1, 11 }, { 1, 3 }, { 3, 0 }, { 1, 205 }, { 9, 255 }, { 1, 152 }, { 64, 0 }, { 1, 28 }, { 3, 0 }, { 1, 48 }, { 10, 255 }, { 1, 143 }, { 63, 0 }, { 1, 25 }, { 1, 18 }, { 3, 0 }, { 1, 147 }, { 10, 255 }, { 1, 61 }, { 62, 0 }, { 1, 4 }, { 1, 52 }, { 3, 0 }, { 1, 7 }, { 1, 238 }, { 9, 255 }, { 1, 206 }, { 63, 0 }, { 1, 36 }, { 1, 35 }, { 3, 0 }, { 1, 89 }, { 9, 255 }, { 1, 249 }, { 1, 46 }, { 62, 0 }, { 1, 8 }, { 1, 61 }, { 1, 9 }, { 3, 0 }, { 1, 188 }, { 9, 255 }, { 1, 116 }, { 63, 0 }, { 1, 38 }, { 1, 41 }, { 3, 0 }, { 1, 32 }, { 9, 255 }, { 1, 192 }, { 1, 2 }, { 62, 0 }, { 1, 8 }, { 1, 61 }, { 1, 11 }, { 3, 0 }, { 1, 129 }, { 8, 255 }, { 1, 240 }, { 1, 30 }, { 63, 0 }, { 1, 39 }, { 1, 44 }, { 3, 0 }, { 1, 2 }, { 1, 226 }, { 8, 255 }, { 1, 92 }, { 5, 0 }, { 1, 48 }, { 1, 144 }, { 1, 166 }, { 1, 125 }, { 1, 23 }, { 53, 0 }, { 1, 8 }, { 1, 61 }, { 1, 15 }, { 3, 0 }, { 1, 71 }, { 8, 255 }, { 1, 169 }, { 4, 0 }, { 1, 1 }, { 1, 119 }, { 1, 251 }, { 3, 255 }, { 1, 241 }, { 1, 38 }, { 52, 0 }, { 1, 36 }, { 1, 46 }, { 4, 0 }, { 1, 170 }, { 7, 255 }, { 1, 228 }, { 1, 17 }, { 3, 0 }, { 1, 15 }, { 1, 176 }, { 6, 255 }, { 1, 200 }, { 1, 3 }, { 50, 0 }, { 1, 2 }, { 1, 58 }, { 1, 11 }, { 3, 0 }, { 1, 20 }, { 1, 249 }, { 7, 255 }, { 1, 68 }, { 3, 0 }, { 1, 45 }, { 1, 219 }, { 8, 255 }, { 1, 121 }, { 50, 0 }, { 1, 23 }, { 1, 37 }, { 4, 0 }, { 1, 112 }, { 7, 255 }, { 1, 144 }, { 3, 0 }, { 1, 92 }, { 1, 245 }, { 9, 255 }, { 1, 249 }, { 1, 47 }, { 49, 0 }, { 1, 43 }, { 1, 5 }, { 4, 0 }, { 1, 211 }, { 6, 255 }, { 1, 213 }, { 1, 8 }, { 1, 0 }, { 1, 6 }, { 1, 149 }, { 12, 255 }, { 1, 209 }, { 1, 6 }, { 47, 0 }, { 1, 9 }, { 1, 21 }, { 4, 0 }, { 1, 54 }, { 6, 255 }, { 1, 250 }, { 1, 49 }, { 1, 0 }, { 1, 29 }, { 1, 200 }, { 14, 255 }, { 1, 133 }, { 47, 0 }, { 1, 14 }, { 5, 0 }, { 1, 153 }, { 6, 255 }, { 1, 144 }, { 1, 0 }, { 1, 68 }, { 1, 235 }, { 15, 255 }, { 1, 252 }, { 1, 56 }, { 51, 0 }, { 1, 10 }, { 1, 241 }, { 6, 255 }, { 1, 204 }, { 1, 136 }, { 1, 253 }, { 17, 255 }, { 1, 217 }, { 1, 9 }, { 50, 0 }, { 1, 95 }, { 28, 255 }, { 1, 145 }, { 50, 0 }, { 1, 194 }, { 29, 255 }, { 1, 57 }, { 48, 0 }, { 1, 37 }, { 30, 255 }, { 1, 156 }, { 48, 0 }, { 1, 136 }, { 30, 255 }, { 1, 124 }, { 47, 0 }, { 1, 3 }, { 1, 230 }, { 29, 255 }, { 1, 221 }, { 1, 27 }, { 47, 0 }, { 1, 77 }, { 28, 255 }, { 1, 231 }, { 1, 143 }, { 1, 24 }, { 48, 0 }, { 1, 161 }, { 20, 255 }, { 1, 251 }, { 1, 220 }, { 1, 186 }, { 1, 153 }, { 1, 119 }, { 1, 85 }, { 1, 51 }, { 1, 18 }, { 51, 0 }, { 1, 180 }, { 13, 255 }, { 1, 234 }, { 1, 200 }, { 1, 167 }, { 1, 133 }, { 1, 99 }, { 1, 65 }, { 1, 32 }, { 1, 3 }, { 58, 0 }, { 1, 79 }, { 1, 252 }, { 4, 255 }, { 1, 247 }, { 1, 214 }, { 1, 181 }, { 1, 147 }, { 1, 113 }, { 1, 79 }, { 1, 46 }, { 1, 13 }, { 67, 0 }, { 1, 48 }, { 1, 102 }, { 1, 93 }, { 1, 59 }, { 1, 26 }, { 1, 1 }, { 112, 0 }, { 1, 19 }, { 1, 22 }, { 76, 0 }, { 1, 18 }, { 1, 49 }, { 1, 25 }, { 75, 0 }, { 1, 7 }, { 1, 43 }, { 1, 56 }, { 1, 16 }, { 75, 0 }, { 1, 27 }, { 1, 59 }, { 1, 40 }, { 1, 4 }, { 74, 0 }, { 1, 4 }, { 1, 47 }, { 1, 56 }, { 1, 19 }, { 56, 0 }, { 1, 1 }, { 1, 7 }, { 1, 1 }, { 16, 0 }, { 1, 11 }, { 1, 46 }, { 1, 25 }, { 1, 1 }, { 56, 0 }, { 1, 8 }, { 1, 12 }, { 17, 0 }, { 1, 9 }, { 1, 19 }, { 57, 0 }, { 1, 5 }, { 1, 17 }, { 1, 11 }, { 76, 0 }, { 1, 11 }, { 1, 19 }, { 1, 7 }, { 75, 0 }, { 1, 4 }, { 1, 17 }, { 1, 16 }, { 1, 3 }, { 75, 0 }, { 1, 10 }, { 1, 20 }, { 1, 11 }, { 75, 0 }, { 1, 2 }, { 1, 16 }, { 1, 18 }, { 1, 5 }, { 75, 0 }, { 1, 4 }, { 1, 18 }, { 1, 10 }, { 1, 1 }, { 75, 0 }, { 1, 7 }, { 1, 12 }, { 1, 2 }, { 76, 0 }, { 1, 5 }, { 1, 4 }, { 852, 0 } }, 
	LogoRLEFrame{ { 1479, 0 }, { 1, 2 }, { 1, 131 }, { 1, 159 }, { 1, 33 }, { 76, 0 }, { 1, 115 }, { 2, 255 }, { 1, 209 }, { 1, 1 }, { 74, 0 }, { 1, 4 }, { 1, 227 }, { 3, 255 }, { 1, 75 }, { 74, 0 }, { 1, 84 }, { 4, 255 }, { 1, 186 }, { 69, 0 }, { 1, 2 }, { 4, 0 }, { 1, 193 }, { 5, 255 }, { 1, 43 }, { 68, 0 }, { 1, 14 }, { 3, 0 }, { 1, 46 }, { 6, 255 }, { 1, 143 }, { 67, 0 }, { 1, 23 }, { 1, 5 }, { 3, 0 }, { 1, 155 }, { 6, 255 }, { 1, 219 }, { 66, 0 }, { 1, 3 }, { 1, 39 }, { 3, 0 }, { 1, 18 }, { 1, 246 }, { 7, 255 }, { 1, 38 }, { 65, 0 }, { 1, 36 }, { 1, 21 }, { 3, 0 }, { 1, 118 }, { 8, 255 }, { 1, 113 }, { 64, 0 }, { 1, 11 }, { 1, 58 }, { 1, 2 }, { 2, 0 }, { 1, 2 }, { 1, 224 }, { 8, 255 }, { 1, 161 }, { 64, 0 }, { 1, 44 }, { 1, 34 }, { 3, 0 }, { 1, 80 }, { 9, 255 }, { 1, 146 }, { 63, 0 }, { 1, 12 }, { 1, 61 }, { 1, 6 }, { 3, 0 }, { 1, 189 }, { 9, 255 }, { 1, 100 }, { 63, 0 }, { 1, 44 }, { 1, 37 }, { 3, 0 }, { 1, 42 }, { 9, 255 }, { 1, 234 }, { 1, 8 }, { 62, 0 }, { 1, 13 }, { 1, 61 }, { 1, 8 }, { 3, 0 }, { 1, 151 }, { 9, 255 }, { 1, 100 }, { 63, 0 }, { 1, 45 }, { 1, 40 }, { 3, 0 }, { 1, 15 }, { 1, 244 }, { 8, 255 }, { 1, 170 }, { 63, 0 }, { 1, 11 }, { 1, 62 }, { 1, 10 }, { 3, 0 }, { 1, 113 }, { 8, 255 }, { 1, 225 }, { 1, 16 }, { 63, 0 }, { 2, 36 }, { 3, 0 }, { 1, 1 }, { 1, 220 }, { 7, 255 }, { 1, 252 }, { 1, 59 }, { 63, 0 }, { 1, 1 }, { 1, 54 }, { 1, 4 }, { 3, 0 }, { 1, 75 }, { 8, 255 }, { 1, 127 }, { 5, 0 }, { 1, 54 }, { 1, 111 }, { 1, 96 }, { 1, 26 }, { 55, 0 }, { 1, 22 }, { 1, 26 }, { 4, 0 }, { 1, 184 }, { 7, 255 }, { 1, 194 }, { 1, 3 }, { 3, 0 }, { 1, 7 }, { 1, 153 }, { 3, 255 }, { 1, 244 }, { 1, 73 }, { 54, 0 }, { 1, 31 }, { 4, 0 }, { 1, 38 }, { 7, 255 }, { 1, 238 }, { 1, 29 }, { 3, 0 }, { 1, 27 }, { 1, 201 }, { 5, 255 }, { 1, 224 }, { 1, 10 }, { 52, 0 }, { 1, 5 }, { 1, 8 }, { 4, 0 }, { 1, 146 }, { 7, 255 }, { 1, 83 }, { 3, 0 }, { 1, 59 }, { 1, 231 }, { 7, 255 }, { 1, 131 }, { 57, 0 }, { 1, 13 }, { 1, 242 }, { 6, 255 }, { 1, 154 }, { 3, 0 }, { 1, 103 }, { 1, 250 }, { 8, 255 }, { 1, 248 }, { 1, 37 }, { 56, 0 }, { 1, 108 }, { 6, 255 }, { 1, 214 }, { 1, 10 }, { 1, 0 }, { 1, 6 }, { 1, 153 }, { 11, 255 }, { 1, 184 }, { 55, 0 }, { 1, 1 }, { 1, 216 }, { 5, 255 }, { 1, 248 }, { 1, 47 }, { 1, 0 }, { 1, 24 }, { 1, 197 }, { 13, 255 }, { 1, 80 }, { 54, 0 }, { 1, 70 }, { 6, 255 }, { 1, 110 }, { 1, 0 }, { 1, 55 }, { 1, 229 }, { 14, 255 }, { 1, 224 }, { 1, 10 }, { 53, 0 }, { 1, 179 }, { 5, 255 }, { 1, 250 }, { 1, 6 }, { 1, 99 }, { 1, 248 }, { 16, 255 }, { 1, 131 }, { 52, 0 }, { 1, 34 }, { 1, 253 }, { 6, 255 }, { 1, 228 }, { 18, 255 }, { 1, 248 }, { 1, 37 }, { 51, 0 }, { 1, 141 }, { 27, 255 }, { 1, 181 }, { 50, 0 }, { 1, 11 }, { 1, 239 }, { 28, 255 }, { 1, 33 }, { 49, 0 }, { 1, 103 }, { 28, 255 }, { 1, 249 }, { 1, 18 }, { 49, 0 }, { 1, 212 }, { 28, 255 }, { 1, 142 }, { 49, 0 }, { 1, 66 }, { 27, 255 }, { 1, 221 }, { 1, 110 }, { 50, 0 }, { 1, 175 }, { 21, 255 }, { 1, 231 }, { 1, 192 }, { 1, 154 }, { 1, 116 }, { 1, 77 }, { 1, 39 }, { 1, 1 }, { 50, 0 }, { 1, 2 }, { 1, 249 }, { 14, 255 }, { 1, 243 }, { 1, 205 }, { 1, 167 }, { 1, 128 }, { 1, 90 }, { 1, 51 }, { 1, 14 }, { 57, 0 }, { 1, 7 }, { 1, 241 }, { 7, 255 }, { 1, 251 }, { 1, 218 }, { 1, 179 }, { 1, 141 }, { 1, 102 }, { 1, 64 }, { 1, 26 }, { 65, 0 }, { 1, 83 }, { 1, 215 }, { 1, 225 }, { 1, 192 }, { 1, 153 }, { 1, 115 }, { 1, 77 }, { 1, 38 }, { 1, 5 }, { 33, 0 }, { 1, 4 }, { 1, 3 }, { 76, 0 }, { 1, 4 }, { 1, 18 }, { 1, 11 }, { 75, 0 }, { 1, 1 }, { 1, 17 }, { 1, 28 }, { 1, 8 }, { 56, 0 }, { 1, 11 }, { 1, 1 }, { 17, 0 }, { 1, 9 }, { 1, 27 }, { 1, 22 }, { 1, 3 }, { 55, 0 }, { 1, 10 }, { 1, 31 }, { 1, 5 }, { 16, 0 }, { 1, 2 }, { 1, 21 }, { 1, 29 }, { 1, 12 }, { 55, 0 }, { 1, 3 }, { 1, 35 }, { 1, 41 }, { 1, 2 }, { 16, 0 }, { 1, 5 }, { 1, 25 }, { 1, 19 }, { 1, 3 }, { 55, 0 }, { 1, 20 }, { 1, 57 }, { 1, 34 }, { 17, 0 }, { 1, 8 }, { 1, 15 }, { 1, 3 }, { 55, 0 }, { 1, 4 }, { 1, 41 }, { 1, 58 }, { 1, 20 }, { 17, 0 }, { 1, 1 }, { 1, 2 }, { 56, 0 }, { 1, 18 }, { 1, 56 }, { 1, 47 }, { 1, 7 }, { 74, 0 }, { 1, 2 }, { 1, 38 }, { 1, 61 }, { 1, 30 }, { 75, 0 }, { 1, 6 }, { 2, 50 }, { 1, 13 }, { 75, 0 }, { 1, 13 }, { 1, 49 }, { 1, 23 }, { 76, 0 }, { 1, 18 }, { 1, 29 }, { 1, 2 }, { 76, 0 }, { 1, 8 }, { 1, 6 }, { 1088, 0 } }, 
	LogoRLEFrame{ { 1396, 0 }, { 1, 1 }, { 78, 0 }, { 1, 14 }, { 78, 0 }, { 1, 3 }, { 1, 25 }, { 5, 0 }, { 1, 82 }, { 1, 174 }, { 1, 71 }, { 70, 0 }, { 1, 35 }, { 1, 7 }, { 4, 0 }, { 1, 43 }, { 1, 250 }, { 1, 255 }, { 1, 243 }, { 1, 18 }, { 68, 0 }, { 1, 11 }, { 1, 46 }, { 5, 0 }, { 1, 164 }, { 3, 255 }, { 1, 109 }, { 68, 0 }, { 1, 47 }, { 1, 24 }, { 4, 0 }, { 1, 33 }, { 1, 252 }, { 3, 255 }, { 1, 209 }, { 67, 0 }, { 1, 18 }, { 1, 57 }, { 1, 2 }, { 4, 0 }, { 1, 151 }, { 5, 255 }, { 1, 51 }, { 66, 0 }, { 1, 49 }, { 1, 30 }, { 4, 0 }, { 1, 24 }, { 1, 248 }, { 5, 255 }, { 1, 124 }, { 65, 0 }, { 1, 18 }, { 1, 59 }, { 1, 3 }, { 4, 0 }, { 1, 138 }, { 6, 255 }, { 1, 187 }, { 65, 0 }, { 1, 49 }, { 1, 33 }, { 4, 0 }, { 1, 16 }, { 1, 242 }, { 6, 255 }, { 1, 245 }, { 1, 5 }, { 63, 0 }, { 1, 18 }, { 1, 60 }, { 1, 5 }, { 4, 0 }, { 1, 124 }, { 8, 255 }, { 1, 40 }, { 63, 0 }, { 1, 48 }, { 1, 34 }, { 4, 0 }, { 1, 10 }, { 1, 235 }, { 8, 255 }, { 1, 15 }, { 62, 0 }, { 1, 9 }, { 1, 56 }, { 1, 3 }, { 4, 0 }, { 1, 111 }, { 8, 255 }, { 1, 214 }, { 63, 0 }, { 1, 34 }, { 1, 20 }, { 4, 0 }, { 1, 5 }, { 1, 226 }, { 8, 255 }, { 1, 85 }, { 62, 0 }, { 1, 1 }, { 1, 39 }, { 5, 0 }, { 1, 98 }, { 8, 255 }, { 1, 182 }, { 1, 1 }, { 62, 0 }, { 1, 17 }, { 1, 8 }, { 4, 0 }, { 1, 2 }, { 1, 217 }, { 7, 255 }, { 1, 227 }, { 1, 20 }, { 63, 0 }, { 1, 11 }, { 5, 0 }, { 1, 84 }, { 7, 255 }, { 1, 251 }, { 1, 60 }, { 70, 0 }, { 1, 205 }, { 7, 255 }, { 1, 119 }, { 70, 0 }, { 1, 71 }, { 7, 255 }, { 1, 182 }, { 1, 1 }, { 3, 0 }, { 1, 22 }, { 1, 153 }, { 1, 209 }, { 1, 187 }, { 1, 82 }, { 62, 0 }, { 1, 192 }, { 6, 255 }, { 1, 227 }, { 1, 20 }, { 3, 0 }, { 1, 56 }, { 1, 232 }, { 4, 255 }, { 1, 58 }, { 60, 0 }, { 1, 58 }, { 6, 255 }, { 1, 251 }, { 1, 60 }, { 3, 0 }, { 1, 91 }, { 1, 248 }, { 5, 255 }, { 1, 187 }, { 60, 0 }, { 1, 179 }, { 6, 255 }, { 1, 119 }, { 2, 0 }, { 1, 1 }, { 1, 133 }, { 8, 255 }, { 1, 62 }, { 58, 0 }, { 1, 46 }, { 6, 255 }, { 1, 182 }, { 1, 1 }, { 1, 0 }, { 1, 11 }, { 1, 174 }, { 9, 255 }, { 1, 192 }, { 58, 0 }, { 1, 166 }, { 5, 255 }, { 1, 227 }, { 1, 20 }, { 1, 0 }, { 1, 28 }, { 1, 207 }, { 11, 255 }, { 1, 65 }, { 56, 0 }, { 1, 34 }, { 1, 252 }, { 4, 255 }, { 1, 251 }, { 1, 59 }, { 1, 0 }, { 1, 55 }, { 1, 231 }, { 12, 255 }, { 1, 195 }, { 56, 0 }, { 1, 152 }, { 5, 255 }, { 1, 119 }, { 1, 0 }, { 1, 89 }, { 1, 247 }, { 14, 255 }, { 1, 68 }, { 54, 0 }, { 1, 25 }, { 1, 248 }, { 4, 255 }, { 1, 227 }, { 1, 2 }, { 1, 131 }, { 16, 255 }, { 1, 198 }, { 54, 0 }, { 1, 139 }, { 6, 255 }, { 1, 216 }, { 18, 255 }, { 1, 71 }, { 52, 0 }, { 1, 17 }, { 1, 243 }, { 25, 255 }, { 1, 196 }, { 52, 0 }, { 1, 126 }, { 26, 255 }, { 1, 239 }, { 51, 0 }, { 1, 11 }, { 1, 236 }, { 26, 255 }, { 1, 153 }, { 51, 0 }, { 1, 112 }, { 26, 255 }, { 1, 183 }, { 1, 11 }, { 50, 0 }, { 1, 6 }, { 1, 228 }, { 22, 255 }, { 1, 246 }, { 1, 203 }, { 1, 155 }, { 1, 65 }, { 1, 1 }, { 51, 0 }, { 1, 99 }, { 17, 255 }, { 1, 253 }, { 1, 218 }, { 1, 173 }, { 1, 128 }, { 1, 83 }, { 1, 37 }, { 1, 2 }, { 55, 0 }, { 1, 202 }, { 12, 255 }, { 1, 233 }, { 1, 188 }, { 1, 142 }, { 1, 97 }, { 1, 52 }, { 1, 10 }, { 61, 0 }, { 1, 232 }, { 6, 255 }, { 1, 246 }, { 1, 202 }, { 1, 157 }, { 1, 112 }, { 1, 67 }, { 1, 22 }, { 10, 0 }, { 1, 1 }, { 56, 0 }, { 1, 94 }, { 1, 219 }, { 1, 214 }, { 1, 172 }, { 1, 127 }, { 1, 82 }, { 1, 37 }, { 1, 2 }, { 13, 0 }, { 1, 1 }, { 1, 20 }, { 1, 6 }, { 76, 0 }, { 1, 18 }, { 1, 39 }, { 1, 5 }, { 75, 0 }, { 1, 9 }, { 1, 44 }, { 1, 42 }, { 1, 2 }, { 75, 0 }, { 1, 27 }, { 1, 60 }, { 1, 32 }, { 75, 0 }, { 1, 7 }, { 1, 46 }, { 1, 56 }, { 1, 16 }, { 75, 0 }, { 1, 24 }, { 1, 59 }, { 1, 43 }, { 1, 4 }, { 74, 0 }, { 1, 2 }, { 1, 42 }, { 1, 60 }, { 1, 25 }, { 75, 0 }, { 1, 6 }, { 1, 50 }, { 1, 41 }, { 1, 8 }, { 75, 0 }, { 1, 13 }, { 1, 43 }, { 1, 14 }, { 76, 0 }, { 1, 14 }, { 1, 19 }, { 77, 0 }, { 2, 1 }, { 1324, 0 } }, 
	LogoRLEFrame{ { 998, 0 }, { 1, 2 }, { 78, 0 }, { 1, 3 }, { 1, 10 }, { 78, 0 }, { 1, 28 }, { 78, 0 }, { 1, 11 }, { 1, 32 }, { 78, 0 }, { 1, 47 }, { 1, 10 }, { 77, 0 }, { 1, 21 }, { 1, 49 }, { 78, 0 }, { 1, 54 }, { 1, 22 }, { 77, 0 }, { 1, 23 }, { 1, 54 }, { 78, 0 }, { 1, 54 }, { 1, 26 }, { 6, 0 }, { 1, 21 }, { 1, 145 }, { 1, 100 }, { 68, 0 }, { 1, 24 }, { 1, 57 }, { 1, 1 }, { 6, 0 }, { 1, 192 }, { 2, 255 }, { 1, 46 }, { 66, 0 }, { 1, 1 }, { 1, 54 }, { 1, 29 }, { 6, 0 }, { 1, 79 }, { 3, 255 }, { 1, 133 }, { 66, 0 }, { 1, 22 }, { 1, 57 }, { 1, 3 }, { 5, 0 }, { 1, 2 }, { 1, 211 }, { 3, 255 }, { 1, 217 }, { 66, 0 }, { 1, 47 }, { 1, 22 }, { 6, 0 }, { 1, 93 }, { 5, 255 }, { 1, 38 }, { 64, 0 }, { 1, 8 }, { 1, 46 }, { 6, 0 }, { 1, 5 }, { 1, 222 }, { 5, 255 }, { 1, 87 }, { 64, 0 }, { 1, 31 }, { 1, 9 }, { 6, 0 }, { 1, 107 }, { 6, 255 }, { 1, 133 }, { 64, 0 }, { 1, 25 }, { 6, 0 }, { 1, 10 }, { 1, 231 }, { 6, 255 }, { 1, 174 }, { 63, 0 }, { 1, 9 }, { 1, 1 }, { 6, 0 }, { 1, 120 }, { 7, 255 }, { 1, 142 }, { 70, 0 }, { 1, 16 }, { 1, 238 }, { 7, 255 }, { 1, 74 }, { 70, 0 }, { 1, 134 }, { 7, 255 }, { 1, 185 }, { 70, 0 }, { 1, 24 }, { 1, 245 }, { 6, 255 }, { 1, 237 }, { 1, 32 }, { 70, 0 }, { 1, 148 }, { 6, 255 }, { 1, 253 }, { 1, 73 }, { 70, 0 }, { 1, 32 }, { 1, 249 }, { 6, 255 }, { 1, 127 }, { 71, 0 }, { 1, 161 }, { 6, 255 }, { 1, 182 }, { 1, 1 }, { 3, 0 }, { 1, 15 }, { 1, 54 }, { 1, 23 }, { 64, 0 }, { 1, 43 }, { 1, 253 }, { 5, 255 }, { 1, 222 }, { 1, 17 }, { 3, 0 }, { 1, 108 }, { 1, 245 }, { 1, 255 }, { 1, 253 }, { 1, 122 }, { 63, 0 }, { 1, 175 }, { 5, 255 }, { 1, 247 }, { 1, 49 }, { 2, 0 }, { 1, 3 }, { 1, 151 }, { 4, 255 }, { 1, 246 }, { 1, 17 }, { 61, 0 }, { 1, 55 }, { 6, 255 }, { 1, 96 }, { 2, 0 }, { 1, 12 }, { 1, 182 }, { 6, 255 }, { 1, 112 }, { 61, 0 }, { 1, 189 }, { 5, 255 }, { 1, 152 }, { 2, 0 }, { 1, 27 }, { 1, 207 }, { 7, 255 }, { 1, 216 }, { 60, 0 }, { 1, 67 }, { 5, 255 }, { 1, 202 }, { 1, 7 }, { 1, 0 }, { 1, 46 }, { 1, 227 }, { 9, 255 }, { 1, 65 }, { 59, 0 }, { 1, 201 }, { 4, 255 }, { 1, 235 }, { 1, 30 }, { 1, 0 }, { 1, 71 }, { 1, 242 }, { 10, 255 }, { 1, 169 }, { 58, 0 }, { 1, 81 }, { 4, 255 }, { 1, 253 }, { 1, 69 }, { 1, 0 }, { 1, 101 }, { 1, 251 }, { 11, 255 }, { 1, 250 }, { 1, 23 }, { 56, 0 }, { 1, 2 }, { 1, 213 }, { 4, 255 }, { 1, 122 }, { 1, 1 }, { 1, 135 }, { 14, 255 }, { 1, 122 }, { 56, 0 }, { 1, 95 }, { 4, 255 }, { 1, 199 }, { 1, 9 }, { 1, 168 }, { 15, 255 }, { 1, 224 }, { 1, 2 }, { 54, 0 }, { 1, 6 }, { 1, 223 }, { 4, 255 }, { 1, 239 }, { 1, 212 }, { 17, 255 }, { 1, 75 }, { 54, 0 }, { 1, 108 }, { 24, 255 }, { 1, 158 }, { 53, 0 }, { 1, 11 }, { 1, 232 }, { 24, 255 }, { 1, 113 }, { 53, 0 }, { 1, 122 }, { 24, 255 }, { 1, 200 }, { 1, 11 }, { 52, 0 }, { 1, 17 }, { 1, 239 }, { 22, 255 }, { 1, 237 }, { 1, 137 }, { 1, 9 }, { 53, 0 }, { 1, 136 }, { 19, 255 }, { 1, 216 }, { 1, 163 }, { 1, 110 }, { 1, 56 }, { 1, 7 }, { 1, 0 }, { 1, 8 }, { 1, 2 }, { 51, 0 }, { 1, 25 }, { 1, 245 }, { 14, 255 }, { 1, 227 }, { 1, 174 }, { 1, 120 }, { 1, 67 }, { 1, 15 }, { 4, 0 }, { 1, 4 }, { 1, 29 }, { 1, 9 }, { 52, 0 }, { 1, 144 }, { 10, 255 }, { 1, 237 }, { 1, 184 }, { 1, 131 }, { 1, 78 }, { 1, 25 }, { 8, 0 }, { 1, 27 }, { 1, 45 }, { 1, 5 }, { 53, 0 }, { 1, 215 }, { 5, 255 }, { 1, 245 }, { 1, 195 }, { 1, 142 }, { 1, 88 }, { 1, 35 }, { 11, 0 }, { 1, 13 }, { 1, 51 }, { 1, 43 }, { 1, 2 }, { 54, 0 }, { 1, 117 }, { 1, 227 }, { 1, 205 }, { 1, 152 }, { 1, 99 }, { 1, 46 }, { 1, 3 }, { 13, 0 }, { 1, 1 }, { 1, 32 }, { 1, 60 }, { 1, 29 }, { 75, 0 }, { 1, 11 }, { 1, 50 }, { 1, 53 }, { 1, 13 }, { 75, 0 }, { 1, 29 }, { 1, 61 }, { 1, 38 }, { 1, 2 }, { 74, 0 }, { 1, 2 }, { 1, 42 }, { 1, 57 }, { 1, 20 }, { 75, 0 }, { 1, 7 }, { 1, 50 }, { 1, 36 }, { 1, 3 }, { 75, 0 }, { 1, 14 }, { 1, 39 }, { 1, 12 }, { 76, 0 }, { 1, 7 }, { 1, 13 }, { 1639, 0 } }, 
	LogoRLEFrame{ { 680, 0 }, { 1, 7 }, { 78, 0 }, { 1, 5 }, { 1, 8 }, { 78, 0 }, { 1, 20 }, { 78, 0 }, { 1, 10 }, { 1, 17 }, { 77, 0 }, { 1, 1 }, { 1, 27 }, { 1, 6 }, { 77, 0 }, { 1, 14 }, { 1, 23 }, { 77, 0 }, { 1, 1 }, { 1, 28 }, { 1, 9 }, { 77, 0 }, { 1, 14 }, { 1, 25 }, { 77, 0 }, { 1, 1 }, { 1, 28 }, { 1, 10 }, { 77, 0 }, { 1, 15 }, { 1, 26 }, { 78, 0 }, { 1, 28 }, { 1, 11 }, { 77, 0 }, { 1, 10 }, { 1, 23 }, { 78, 0 }, { 1, 21 }, { 1, 5 }, { 7, 0 }, { 1, 85 }, { 1, 118 }, { 1, 1 }, { 67, 0 }, { 1, 3 }, { 1, 16 }, { 7, 0 }, { 1, 93 }, { 2, 255 }, { 1, 79 }, { 67, 0 }, { 1, 11 }, { 1, 1 }, { 6, 0 }, { 1, 16 }, { 1, 235 }, { 2, 255 }, { 1, 149 }, { 67, 0 }, { 1, 5 }, { 7, 0 }, { 1, 144 }, { 3, 255 }, { 1, 216 }, { 74, 0 }, { 1, 41 }, { 1, 251 }, { 4, 255 }, { 1, 10 }, { 73, 0 }, { 1, 185 }, { 5, 255 }, { 1, 38 }, { 72, 0 }, { 1, 78 }, { 6, 255 }, { 1, 67 }, { 71, 0 }, { 1, 6 }, { 1, 219 }, { 6, 255 }, { 1, 37 }, { 71, 0 }, { 1, 119 }, { 6, 255 }, { 1, 212 }, { 71, 0 }, { 1, 24 }, { 1, 242 }, { 5, 255 }, { 1, 253 }, { 1, 60 }, { 71, 0 }, { 1, 160 }, { 6, 255 }, { 1, 129 }, { 71, 0 }, { 1, 54 }, { 6, 255 }, { 1, 177 }, { 1, 1 }, { 70, 0 }, { 1, 1 }, { 1, 200 }, { 5, 255 }, { 1, 214 }, { 1, 13 }, { 71, 0 }, { 1, 93 }, { 5, 255 }, { 1, 239 }, { 1, 37 }, { 3, 0 }, { 1, 77 }, { 1, 130 }, { 1, 91 }, { 1, 2 }, { 64, 0 }, { 1, 12 }, { 1, 229 }, { 4, 255 }, { 1, 253 }, { 1, 72 }, { 2, 0 }, { 1, 11 }, { 1, 177 }, { 3, 255 }, { 1, 131 }, { 64, 0 }, { 1, 134 }, { 5, 255 }, { 1, 118 }, { 2, 0 }, { 1, 22 }, { 1, 203 }, { 4, 255 }, { 1, 217 }, { 63, 0 }, { 1, 34 }, { 1, 248 }, { 4, 255 }, { 1, 167 }, { 2, 0 }, { 1, 36 }, { 1, 220 }, { 6, 255 }, { 1, 42 }, { 62, 0 }, { 1, 175 }, { 4, 255 }, { 1, 206 }, { 1, 9 }, { 1, 0 }, { 1, 53 }, { 1, 234 }, { 7, 255 }, { 1, 123 }, { 51, 0 }, { 1, 4 }, { 9, 0 }, { 1, 68 }, { 4, 255 }, { 1, 234 }, { 1, 30 }, { 1, 0 }, { 1, 74 }, { 1, 244 }, { 8, 255 }, { 1, 204 }, { 50, 0 }, { 1, 1 }, { 1, 9 }, { 8, 0 }, { 1, 4 }, { 1, 212 }, { 3, 255 }, { 1, 250 }, { 1, 62 }, { 1, 0 }, { 1, 98 }, { 1, 251 }, { 10, 255 }, { 1, 30 }, { 49, 0 }, { 1, 13 }, { 1, 2 }, { 8, 0 }, { 1, 109 }, { 4, 255 }, { 1, 106 }, { 1, 0 }, { 1, 125 }, { 12, 255 }, { 1, 110 }, { 48, 0 }, { 1, 7 }, { 1, 13 }, { 8, 0 }, { 1, 19 }, { 1, 238 }, { 3, 255 }, { 1, 155 }, { 1, 3 }, { 1, 153 }, { 13, 255 }, { 1, 191 }, { 47, 0 }, { 1, 2 }, { 1, 19 }, { 1, 4 }, { 8, 0 }, { 1, 150 }, { 3, 255 }, { 1, 204 }, { 1, 16 }, { 1, 177 }, { 14, 255 }, { 1, 252 }, { 1, 20 }, { 46, 0 }, { 2, 13 }, { 8, 0 }, { 1, 46 }, { 1, 252 }, { 3, 255 }, { 1, 213 }, { 1, 206 }, { 16, 255 }, { 1, 89 }, { 45, 0 }, { 1, 4 }, { 1, 20 }, { 1, 2 }, { 8, 0 }, { 1, 190 }, { 22, 255 }, { 1, 116 }, { 45, 0 }, { 1, 17 }, { 1, 11 }, { 8, 0 }, { 1, 83 }, { 22, 255 }, { 1, 243 }, { 1, 24 }, { 44, 0 }, { 1, 8 }, { 1, 19 }, { 1, 1 }, { 7, 0 }, { 1, 8 }, { 1, 223 }, { 21, 255 }, { 1, 231 }, { 1, 53 }, { 5, 0 }, { 1, 16 }, { 1, 8 }, { 37, 0 }, { 1, 1 }, { 1, 19 }, { 1, 8 }, { 8, 0 }, { 1, 124 }, { 19, 255 }, { 1, 247 }, { 1, 191 }, { 1, 108 }, { 1, 11 }, { 4, 0 }, { 1, 10 }, { 1, 38 }, { 1, 10 }, { 38, 0 }, { 1, 10 }, { 1, 16 }, { 8, 0 }, { 1, 28 }, { 1, 245 }, { 15, 255 }, { 1, 241 }, { 1, 182 }, { 1, 120 }, { 1, 59 }, { 1, 6 }, { 5, 0 }, { 1, 3 }, { 1, 35 }, { 1, 49 }, { 1, 6 }, { 38, 0 }, { 1, 1 }, { 1, 18 }, { 1, 3 }, { 8, 0 }, { 1, 165 }, { 12, 255 }, { 1, 234 }, { 1, 173 }, { 1, 111 }, { 1, 50 }, { 1, 3 }, { 8, 0 }, { 1, 17 }, { 1, 56 }, { 1, 42 }, { 1, 2 }, { 39, 0 }, { 2, 9 }, { 8, 0 }, { 1, 58 }, { 9, 255 }, { 1, 226 }, { 1, 164 }, { 1, 102 }, { 1, 40 }, { 11, 0 }, { 1, 2 }, { 1, 37 }, { 1, 60 }, { 1, 25 }, { 41, 0 }, { 1, 12 }, { 9, 0 }, { 1, 168 }, { 5, 255 }, { 1, 216 }, { 1, 155 }, { 1, 93 }, { 1, 31 }, { 14, 0 }, { 1, 15 }, { 1, 54 }, { 1, 50 }, { 1, 9 }, { 41, 0 }, { 1, 5 }, { 1, 1 }, { 9, 0 }, { 1, 133 }, { 1, 243 }, { 1, 207 }, { 1, 146 }, { 1, 84 }, { 1, 23 }, { 17, 0 }, { 1, 32 }, { 1, 62 }, { 1, 34 }, { 1, 1 }, { 42, 0 }, { 1, 1 }, { 31, 0 }, { 1, 2 }, { 1, 43 }, { 1, 49 }, { 1, 14 }, { 75, 0 }, { 1, 7 }, { 1, 43 }, { 1, 22 }, { 76, 0 }, { 1, 10 }, { 1, 24 }, { 1, 1 }, { 77, 0 }, { 1, 3 }, { 1875, 0 } }, 
	LogoRLEFrame{ { 1643, 0 }, { 1, 62 }, { 1, 116 }, { 1, 1 }, { 76, 0 }, { 1, 66 }, { 1, 251 }, { 1, 255 }, { 1, 64 }, { 75, 0 }, { 1, 10 }, { 1, 223 }, { 2, 255 }, { 1, 116 }, { 75, 0 }, { 1, 138 }, { 3, 255 }, { 1, 165 }, { 74, 0 }, { 1, 47 }, { 1, 251 }, { 3, 255 }, { 1, 184 }, { 73, 0 }, { 1, 2 }, { 1, 200 }, { 4, 255 }, { 1, 195 }, { 73, 0 }, { 1, 107 }, { 5, 255 }, { 1, 180 }, { 72, 0 }, { 1, 26 }, { 1, 241 }, { 5, 255 }, { 1, 93 }, { 72, 0 }, { 1, 173 }, { 5, 255 }, { 1, 194 }, { 1, 3 }, { 71, 0 }, { 1, 77 }, { 5, 255 }, { 1, 231 }, { 1, 26 }, { 71, 0 }, { 1, 12 }, { 1, 226 }, { 4, 255 }, { 1, 247 }, { 1, 54 }, { 72, 0 }, { 1, 143 }, { 5, 255 }, { 1, 89 }, { 72, 0 }, { 1, 50 }, { 1, 252 }, { 4, 255 }, { 1, 129 }, { 2, 0 }, { 1, 3 }, { 1, 102 }, { 1, 161 }, { 1, 117 }, { 1, 5 }, { 56, 0 }, { 1, 5 }, { 8, 0 }, { 1, 3 }, { 1, 204 }, { 4, 255 }, { 1, 170 }, { 1, 1 }, { 1, 0 }, { 1, 17 }, { 1, 194 }, { 3, 255 }, { 1, 100 }, { 55, 0 }, { 1, 4 }, { 1, 16 }, { 8, 0 }, { 1, 112 }, { 4, 255 }, { 1, 204 }, { 1, 9 }, { 1, 0 }, { 1, 27 }, { 1, 212 }, { 4, 255 }, { 1, 161 }, { 55, 0 }, { 1, 34 }, { 8, 0 }, { 1, 29 }, { 1, 243 }, { 3, 255 }, { 1, 229 }, { 1, 26 }, { 1, 0 }, { 1, 40 }, { 1, 225 }, { 5, 255 }, { 1, 223 }, { 54, 0 }, { 1, 22 }, { 1, 28 }, { 8, 0 }, { 1, 178 }, { 3, 255 }, { 1, 246 }, { 1, 52 }, { 1, 0 }, { 1, 56 }, { 1, 236 }, { 7, 255 }, { 1, 29 }, { 52, 0 }, { 1, 7 }, { 1, 55 }, { 1, 3 }, { 7, 0 }, { 1, 82 }, { 4, 255 }, { 1, 85 }, { 1, 0 }, { 1, 74 }, { 1, 245 }, { 8, 255 }, { 1, 90 }, { 52, 0 }, { 1, 45 }, { 1, 34 }, { 7, 0 }, { 1, 14 }, { 1, 229 }, { 3, 255 }, { 1, 125 }, { 1, 0 }, { 1, 94 }, { 1, 251 }, { 9, 255 }, { 1, 152 }, { 51, 0 }, { 1, 20 }, { 1, 58 }, { 1, 4 }, { 7, 0 }, { 1, 148 }, { 3, 255 }, { 1, 167 }, { 1, 0 }, { 1, 117 }, { 11, 255 }, { 1, 213 }, { 50, 0 }, { 1, 2 }, { 1, 55 }, { 1, 26 }, { 7, 0 }, { 1, 55 }, { 1, 253 }, { 2, 255 }, { 1, 201 }, { 1, 10 }, { 1, 141 }, { 13, 255 }, { 1, 20 }, { 49, 0 }, { 1, 31 }, { 1, 53 }, { 7, 0 }, { 1, 4 }, { 1, 208 }, { 2, 255 }, { 1, 227 }, { 1, 30 }, { 1, 164 }, { 14, 255 }, { 1, 80 }, { 48, 0 }, { 1, 8 }, { 1, 61 }, { 1, 18 }, { 7, 0 }, { 1, 117 }, { 3, 255 }, { 1, 177 }, { 1, 187 }, { 15, 255 }, { 1, 129 }, { 11, 0 }, { 1, 4 }, { 1, 3 }, { 35, 0 }, { 1, 42 }, { 1, 46 }, { 7, 0 }, { 1, 32 }, { 1, 245 }, { 20, 255 }, { 1, 116 }, { 9, 0 }, { 1, 1 }, { 1, 25 }, { 1, 14 }, { 35, 0 }, { 1, 10 }, { 1, 59 }, { 1, 8 }, { 7, 0 }, { 1, 183 }, { 20, 255 }, { 1, 212 }, { 1, 9 }, { 8, 0 }, { 1, 18 }, { 1, 46 }, { 1, 11 }, { 36, 0 }, { 1, 40 }, { 1, 26 }, { 7, 0 }, { 1, 87 }, { 20, 255 }, { 1, 171 }, { 1, 15 }, { 7, 0 }, { 1, 6 }, { 1, 44 }, { 1, 50 }, { 1, 6 }, { 36, 0 }, { 1, 7 }, { 1, 45 }, { 7, 0 }, { 1, 16 }, { 1, 231 }, { 16, 255 }, { 1, 251 }, { 1, 195 }, { 1, 126 }, { 1, 40 }, { 8, 0 }, { 1, 23 }, { 1, 58 }, { 1, 38 }, { 1, 2 }, { 37, 0 }, { 1, 30 }, { 1, 5 }, { 7, 0 }, { 1, 153 }, { 14, 255 }, { 1, 217 }, { 1, 147 }, { 1, 78 }, { 1, 14 }, { 9, 0 }, { 1, 5 }, { 1, 43 }, { 1, 58 }, { 1, 20 }, { 38, 0 }, { 1, 4 }, { 1, 12 }, { 7, 0 }, { 1, 59 }, { 11, 255 }, { 1, 236 }, { 1, 168 }, { 1, 99 }, { 1, 30 }, { 12, 0 }, { 1, 19 }, { 1, 57 }, { 1, 47 }, { 1, 7 }, { 39, 0 }, { 1, 1 }, { 7, 0 }, { 1, 5 }, { 1, 212 }, { 7, 255 }, { 1, 249 }, { 1, 190 }, { 1, 120 }, { 1, 51 }, { 1, 2 }, { 14, 0 }, { 1, 33 }, { 1, 60 }, { 1, 29 }, { 49, 0 }, { 1, 106 }, { 5, 255 }, { 1, 211 }, { 1, 141 }, { 1, 72 }, { 1, 10 }, { 16, 0 }, { 1, 2 }, { 1, 43 }, { 1, 39 }, { 1, 6 }, { 50, 0 }, { 1, 156 }, { 1, 255 }, { 1, 231 }, { 1, 163 }, { 1, 93 }, { 1, 25 }, { 19, 0 }, { 1, 7 }, { 1, 36 }, { 1, 12 }, { 52, 0 }, { 1, 4 }, { 1, 24 }, { 22, 0 }, { 1, 5 }, { 1, 14 }, { 2190, 0 } }, 
	LogoRLEFrame{ { 1724, 0 }, { 1, 99 }, { 1, 81 }, { 77, 0 }, { 1, 129 }, { 1, 255 }, { 1, 228 }, { 76, 0 }, { 1, 57 }, { 1, 253 }, { 2, 255 }, { 1, 9 }, { 74, 0 }, { 1, 8 }, { 1, 216 }, { 3, 255 }, { 1, 35 }, { 74, 0 }, { 1, 140 }, { 4, 255 }, { 1, 31 }, { 73, 0 }, { 1, 58 }, { 1, 253 }, { 4, 255 }, { 1, 23 }, { 72, 0 }, { 1, 9 }, { 1, 217 }, { 4, 255 }, { 1, 201 }, { 73, 0 }, { 1, 141 }, { 5, 255 }, { 1, 72 }, { 61, 0 }, { 2, 5 }, { 9, 0 }, { 1, 58 }, { 1, 253 }, { 4, 255 }, { 1, 123 }, { 62, 0 }, { 1, 25 }, { 9, 0 }, { 1, 9 }, { 1, 217 }, { 4, 255 }, { 1, 162 }, { 62, 0 }, { 1, 24 }, { 1, 16 }, { 9, 0 }, { 1, 142 }, { 4, 255 }, { 1, 192 }, { 1, 6 }, { 61, 0 }, { 1, 8 }, { 1, 47 }, { 9, 0 }, { 1, 59 }, { 1, 253 }, { 3, 255 }, { 1, 216 }, { 1, 17 }, { 2, 0 }, { 1, 78 }, { 1, 131 }, { 1, 73 }, { 57, 0 }, { 1, 47 }, { 1, 23 }, { 8, 0 }, { 1, 9 }, { 1, 218 }, { 3, 255 }, { 1, 235 }, { 1, 35 }, { 1, 0 }, { 1, 10 }, { 1, 178 }, { 3, 255 }, { 1, 39 }, { 55, 0 }, { 1, 25 }, { 1, 53 }, { 1, 1 }, { 8, 0 }, { 1, 143 }, { 3, 255 }, { 1, 248 }, { 1, 58 }, { 1, 0 }, { 1, 18 }, { 1, 199 }, { 4, 255 }, { 1, 87 }, { 54, 0 }, { 1, 4 }, { 1, 58 }, { 1, 19 }, { 8, 0 }, { 1, 60 }, { 1, 253 }, { 3, 255 }, { 1, 88 }, { 1, 0 }, { 1, 28 }, { 1, 213 }, { 5, 255 }, { 1, 133 }, { 54, 0 }, { 1, 37 }, { 1, 47 }, { 8, 0 }, { 1, 10 }, { 1, 219 }, { 3, 255 }, { 1, 121 }, { 1, 0 }, { 1, 40 }, { 1, 225 }, { 6, 255 }, { 1, 180 }, { 1, 0 }, { 1, 18 }, { 1, 5 }, { 50, 0 }, { 1, 12 }, { 1, 61 }, { 1, 12 }, { 8, 0 }, { 1, 144 }, { 3, 255 }, { 1, 157 }, { 1, 0 }, { 1, 54 }, { 1, 235 }, { 7, 255 }, { 1, 227 }, { 1, 38 }, { 1, 6 }, { 51, 0 }, { 1, 49 }, { 1, 39 }, { 8, 0 }, { 1, 61 }, { 3, 255 }, { 1, 188 }, { 1, 5 }, { 1, 69 }, { 1, 243 }, { 9, 255 }, { 1, 20 }, { 51, 0 }, { 1, 21 }, { 1, 59 }, { 1, 6 }, { 7, 0 }, { 1, 10 }, { 1, 220 }, { 2, 255 }, { 1, 214 }, { 1, 16 }, { 1, 87 }, { 1, 249 }, { 10, 255 }, { 1, 63 }, { 16, 0 }, { 1, 11 }, { 1, 10 }, { 33, 0 }, { 1, 50 }, { 1, 23 }, { 8, 0 }, { 1, 145 }, { 2, 255 }, { 1, 233 }, { 1, 32 }, { 1, 107 }, { 1, 253 }, { 11, 255 }, { 1, 110 }, { 14, 0 }, { 1, 4 }, { 1, 34 }, { 1, 17 }, { 33, 0 }, { 1, 17 }, { 1, 44 }, { 8, 0 }, { 1, 62 }, { 2, 255 }, { 1, 246 }, { 1, 55 }, { 1, 128 }, { 13, 255 }, { 1, 157 }, { 13, 0 }, { 1, 27 }, { 1, 51 }, { 1, 11 }, { 34, 0 }, { 1, 42 }, { 1, 6 }, { 7, 0 }, { 1, 11 }, { 1, 221 }, { 2, 255 }, { 1, 124 }, { 1, 150 }, { 14, 255 }, { 1, 193 }, { 11, 0 }, { 1, 10 }, { 1, 49 }, { 1, 50 }, { 1, 6 }, { 34, 0 }, { 1, 13 }, { 1, 15 }, { 8, 0 }, { 1, 147 }, { 19, 255 }, { 1, 171 }, { 10, 0 }, { 1, 28 }, { 1, 60 }, { 1, 34 }, { 1, 1 }, { 35, 0 }, { 1, 11 }, { 8, 0 }, { 1, 63 }, { 19, 255 }, { 1, 232 }, { 1, 33 }, { 8, 0 }, { 1, 8 }, { 1, 47 }, { 1, 56 }, { 1, 16 }, { 45, 0 }, { 1, 11 }, { 1, 221 }, { 18, 255 }, { 1, 189 }, { 1, 29 }, { 8, 0 }, { 1, 22 }, { 1, 59 }, { 1, 43 }, { 1, 4 }, { 46, 0 }, { 1, 148 }, { 16, 255 }, { 1, 214 }, { 1, 139 }, { 1, 53 }, { 9, 0 }, { 1, 33 }, { 1, 55 }, { 1, 22 }, { 47, 0 }, { 1, 64 }, { 13, 255 }, { 1, 247 }, { 1, 182 }, { 1, 107 }, { 1, 33 }, { 10, 0 }, { 1, 2 }, { 1, 40 }, { 1, 30 }, { 1, 2 }, { 47, 0 }, { 1, 11 }, { 1, 222 }, { 10, 255 }, { 1, 226 }, { 1, 151 }, { 1, 76 }, { 1, 10 }, { 12, 0 }, { 1, 7 }, { 1, 26 }, { 1, 5 }, { 49, 0 }, { 1, 149 }, { 7, 255 }, { 1, 252 }, { 1, 195 }, { 1, 120 }, { 1, 45 }, { 16, 0 }, { 1, 5 }, { 50, 0 }, { 1, 61 }, { 5, 255 }, { 1, 236 }, { 1, 163 }, { 1, 89 }, { 1, 18 }, { 70, 0 }, { 1, 173 }, { 2, 255 }, { 1, 207 }, { 1, 132 }, { 1, 57 }, { 1, 2 }, { 73, 0 }, { 1, 55 }, { 1, 86 }, { 1, 27 }, { 2213, 0 } }, 
	LogoRLEFrame{ { 1804, 0 }, { 1, 34 }, { 1, 136 }, { 1, 20 }, { 76, 0 }, { 1, 26 }, { 1, 229 }, { 1, 255 }, { 1, 94 }, { 63, 0 }, { 1, 1 }, { 11, 0 }, { 1, 1 }, { 1, 185 }, { 2, 255 }, { 1, 116 }, { 63, 0 }, { 1, 14 }, { 11, 0 }, { 1, 108 }, { 3, 255 }, { 1, 124 }, { 62, 0 }, { 1, 23 }, { 1, 6 }, { 10, 0 }, { 1, 40 }, { 1, 246 }, { 3, 255 }, { 1, 104 }, { 61, 0 }, { 1, 8 }, { 1, 36 }, { 10, 0 }, { 1, 5 }, { 1, 204 }, { 4, 255 }, { 1, 67 }, { 61, 0 }, { 1, 48 }, { 1, 11 }, { 10, 0 }, { 1, 132 }, { 4, 255 }, { 1, 209 }, { 61, 0 }, { 1, 29 }, { 1, 46 }, { 10, 0 }, { 1, 58 }, { 1, 252 }, { 3, 255 }, { 1, 241 }, { 1, 44 }, { 60, 0 }, { 1, 7 }, { 1, 60 }, { 1, 13 }, { 9, 0 }, { 1, 12 }, { 1, 221 }, { 3, 255 }, { 1, 251 }, { 1, 71 }, { 61, 0 }, { 1, 42 }, { 1, 41 }, { 10, 0 }, { 1, 157 }, { 4, 255 }, { 1, 98 }, { 61, 0 }, { 1, 17 }, { 1, 60 }, { 1, 7 }, { 9, 0 }, { 1, 79 }, { 4, 255 }, { 1, 128 }, { 2, 0 }, { 1, 6 }, { 1, 38 }, { 1, 6 }, { 56, 0 }, { 1, 1 }, { 1, 53 }, { 1, 32 }, { 9, 0 }, { 1, 23 }, { 1, 234 }, { 3, 255 }, { 1, 158 }, { 2, 0 }, { 1, 85 }, { 1, 235 }, { 1, 255 }, { 1, 221 }, { 1, 4 }, { 5, 0 }, { 1, 9 }, { 1, 3 }, { 48, 0 }, { 1, 28 }, { 1, 57 }, { 1, 3 }, { 9, 0 }, { 1, 180 }, { 3, 255 }, { 1, 185 }, { 1, 4 }, { 1, 0 }, { 1, 114 }, { 4, 255 }, { 1, 35 }, { 3, 0 }, { 1, 4 }, { 1, 30 }, { 1, 10 }, { 48, 0 }, { 1, 2 }, { 1, 58 }, { 1, 22 }, { 9, 0 }, { 1, 104 }, { 3, 255 }, { 1, 207 }, { 1, 13 }, { 1, 0 }, { 1, 136 }, { 5, 255 }, { 1, 71 }, { 2, 0 }, { 1, 23 }, { 1, 47 }, { 1, 7 }, { 17, 0 }, { 1, 1 }, { 1, 2 }, { 30, 0 }, { 1, 27 }, { 1, 43 }, { 9, 0 }, { 1, 37 }, { 1, 245 }, { 2, 255 }, { 1, 225 }, { 1, 25 }, { 1, 3 }, { 1, 157 }, { 6, 255 }, { 1, 107 }, { 1, 8 }, { 1, 47 }, { 1, 48 }, { 1, 4 }, { 17, 0 }, { 2, 9 }, { 30, 0 }, { 1, 1 }, { 1, 51 }, { 1, 5 }, { 8, 0 }, { 1, 4 }, { 1, 201 }, { 2, 255 }, { 1, 239 }, { 1, 42 }, { 1, 8 }, { 1, 176 }, { 7, 255 }, { 1, 152 }, { 1, 59 }, { 1, 36 }, { 1, 1 }, { 16, 0 }, { 1, 5 }, { 1, 21 }, { 1, 9 }, { 31, 0 }, { 1, 24 }, { 1, 18 }, { 9, 0 }, { 1, 128 }, { 2, 255 }, { 1, 249 }, { 1, 64 }, { 1, 15 }, { 1, 193 }, { 8, 255 }, { 1, 195 }, { 1, 20 }, { 16, 0 }, { 1, 1 }, { 1, 17 }, { 1, 27 }, { 1, 5 }, { 32, 0 }, { 1, 23 }, { 9, 0 }, { 1, 55 }, { 1, 251 }, { 2, 255 }, { 1, 89 }, { 1, 24 }, { 1, 208 }, { 9, 255 }, { 1, 215 }, { 16, 0 }, { 1, 7 }, { 1, 26 }, { 1, 23 }, { 1, 3 }, { 32, 0 }, { 1, 4 }, { 9, 0 }, { 1, 11 }, { 1, 218 }, { 2, 255 }, { 1, 118 }, { 1, 35 }, { 1, 221 }, { 10, 255 }, { 1, 248 }, { 1, 2 }, { 14, 0 }, { 1, 16 }, { 1, 30 }, { 1, 14 }, { 44, 0 }, { 1, 152 }, { 2, 255 }, { 1, 149 }, { 1, 48 }, { 1, 232 }, { 12, 255 }, { 1, 29 }, { 12, 0 }, { 1, 6 }, { 1, 25 }, { 1, 26 }, { 1, 6 }, { 44, 0 }, { 1, 76 }, { 2, 255 }, { 1, 178 }, { 1, 66 }, { 1, 241 }, { 13, 255 }, { 1, 62 }, { 11, 0 }, { 1, 11 }, { 1, 30 }, { 1, 19 }, { 1, 1 }, { 44, 0 }, { 1, 21 }, { 1, 232 }, { 2, 255 }, { 1, 185 }, { 1, 248 }, { 14, 255 }, { 1, 59 }, { 10, 0 }, { 1, 16 }, { 1, 23 }, { 1, 6 }, { 46, 0 }, { 1, 176 }, { 18, 255 }, { 1, 184 }, { 9, 0 }, { 1, 1 }, { 1, 17 }, { 1, 10 }, { 47, 0 }, { 1, 99 }, { 18, 255 }, { 1, 168 }, { 1, 10 }, { 8, 0 }, { 1, 1 }, { 1, 8 }, { 48, 0 }, { 1, 34 }, { 1, 243 }, { 15, 255 }, { 1, 242 }, { 1, 169 }, { 1, 59 }, { 59, 0 }, { 1, 3 }, { 1, 198 }, { 13, 255 }, { 1, 227 }, { 1, 149 }, { 1, 71 }, { 1, 6 }, { 61, 0 }, { 1, 123 }, { 11, 255 }, { 1, 206 }, { 1, 128 }, { 1, 50 }, { 64, 0 }, { 1, 51 }, { 1, 250 }, { 7, 255 }, { 1, 250 }, { 1, 185 }, { 1, 107 }, { 1, 30 }, { 66, 0 }, { 1, 9 }, { 1, 215 }, { 5, 255 }, { 1, 238 }, { 1, 164 }, { 1, 86 }, { 1, 14 }, { 69, 0 }, { 1, 131 }, { 3, 255 }, { 1, 222 }, { 1, 143 }, { 1, 65 }, { 1, 4 }, { 72, 0 }, { 1, 155 }, { 1, 191 }, { 1, 122 }, { 1, 44 }, { 2212, 0 } }, 
	LogoRLEFrame{ { 1633, 0 }, { 1, 4 }, { 78, 0 }, { 1, 18 }, { 78, 0 }, { 1, 9 }, { 1, 25 }, { 78, 0 }, { 1, 45 }, { 1, 3 }, { 10, 0 }, { 1, 9 }, { 1, 171 }, { 1, 197 }, { 64, 0 }, { 1, 31 }, { 1, 34 }, { 11, 0 }, { 1, 167 }, { 1, 255 }, { 1, 248 }, { 63, 0 }, { 1, 11 }, { 1, 60 }, { 1, 7 }, { 10, 0 }, { 1, 96 }, { 3, 255 }, { 1, 3 }, { 62, 0 }, { 1, 47 }, { 1, 34 }, { 10, 0 }, { 1, 37 }, { 1, 243 }, { 2, 255 }, { 1, 240 }, { 62, 0 }, { 1, 22 }, { 1, 57 }, { 1, 3 }, { 9, 0 }, { 1, 5 }, { 1, 203 }, { 3, 255 }, { 1, 211 }, { 61, 0 }, { 1, 3 }, { 1, 56 }, { 1, 26 }, { 10, 0 }, { 1, 136 }, { 4, 255 }, { 1, 115 }, { 61, 0 }, { 1, 34 }, { 1, 52 }, { 10, 0 }, { 1, 66 }, { 1, 253 }, { 3, 255 }, { 1, 208 }, { 1, 9 }, { 60, 0 }, { 1, 9 }, { 1, 61 }, { 1, 18 }, { 9, 0 }, { 1, 19 }, { 1, 229 }, { 3, 255 }, { 1, 225 }, { 1, 26 }, { 13, 0 }, { 1, 2 }, { 47, 0 }, { 1, 38 }, { 1, 41 }, { 10, 0 }, { 1, 176 }, { 3, 255 }, { 1, 238 }, { 1, 41 }, { 13, 0 }, { 1, 19 }, { 1, 12 }, { 46, 0 }, { 1, 6 }, { 1, 55 }, { 1, 4 }, { 9, 0 }, { 1, 104 }, { 3, 255 }, { 1, 247 }, { 1, 59 }, { 12, 0 }, { 1, 10 }, { 1, 41 }, { 1, 13 }, { 47, 0 }, { 1, 34 }, { 1, 19 }, { 9, 0 }, { 1, 42 }, { 1, 246 }, { 2, 255 }, { 1, 253 }, { 1, 80 }, { 1, 0 }, { 1, 10 }, { 1, 136 }, { 1, 221 }, { 1, 194 }, { 1, 24 }, { 5, 0 }, { 1, 2 }, { 1, 33 }, { 1, 52 }, { 1, 8 }, { 47, 0 }, { 1, 4 }, { 1, 32 }, { 9, 0 }, { 1, 7 }, { 1, 209 }, { 3, 255 }, { 1, 104 }, { 1, 0 }, { 1, 19 }, { 1, 200 }, { 3, 255 }, { 1, 85 }, { 4, 0 }, { 1, 12 }, { 1, 52 }, { 1, 47 }, { 1, 5 }, { 48, 0 }, { 1, 17 }, { 1, 1 }, { 9, 0 }, { 1, 144 }, { 3, 255 }, { 1, 131 }, { 1, 0 }, { 1, 29 }, { 1, 214 }, { 4, 255 }, { 1, 113 }, { 3, 0 }, { 1, 29 }, { 1, 60 }, { 1, 32 }, { 50, 0 }, { 1, 2 }, { 9, 0 }, { 1, 74 }, { 3, 255 }, { 1, 157 }, { 1, 0 }, { 1, 41 }, { 1, 226 }, { 5, 255 }, { 1, 142 }, { 1, 0 }, { 1, 7 }, { 1, 47 }, { 1, 57 }, { 1, 16 }, { 60, 0 }, { 1, 23 }, { 1, 233 }, { 2, 255 }, { 1, 181 }, { 1, 4 }, { 1, 55 }, { 1, 236 }, { 6, 255 }, { 1, 171 }, { 1, 16 }, { 1, 58 }, { 1, 46 }, { 1, 5 }, { 60, 0 }, { 1, 1 }, { 1, 184 }, { 2, 255 }, { 1, 201 }, { 1, 10 }, { 1, 71 }, { 1, 244 }, { 7, 255 }, { 1, 204 }, { 1, 56 }, { 1, 24 }, { 62, 0 }, { 1, 113 }, { 2, 255 }, { 1, 218 }, { 1, 20 }, { 1, 90 }, { 1, 250 }, { 8, 255 }, { 1, 231 }, { 1, 3 }, { 62, 0 }, { 1, 48 }, { 1, 249 }, { 1, 255 }, { 1, 232 }, { 1, 33 }, { 1, 110 }, { 10, 255 }, { 1, 253 }, { 1, 4 }, { 61, 0 }, { 1, 10 }, { 1, 215 }, { 1, 255 }, { 1, 243 }, { 1, 50 }, { 1, 132 }, { 12, 255 }, { 1, 29 }, { 61, 0 }, { 1, 153 }, { 1, 255 }, { 1, 250 }, { 1, 72 }, { 1, 154 }, { 13, 255 }, { 1, 58 }, { 60, 0 }, { 1, 82 }, { 2, 255 }, { 2, 175 }, { 14, 255 }, { 1, 64 }, { 59, 0 }, { 1, 28 }, { 1, 237 }, { 17, 255 }, { 1, 209 }, { 1, 5 }, { 58, 0 }, { 1, 2 }, { 1, 191 }, { 17, 255 }, { 1, 204 }, { 1, 28 }, { 59, 0 }, { 1, 121 }, { 16, 255 }, { 1, 219 }, { 1, 106 }, { 1, 7 }, { 59, 0 }, { 1, 54 }, { 1, 251 }, { 13, 255 }, { 1, 205 }, { 1, 125 }, { 1, 45 }, { 61, 0 }, { 1, 13 }, { 1, 220 }, { 10, 255 }, { 1, 252 }, { 1, 190 }, { 1, 110 }, { 1, 30 }, { 64, 0 }, { 1, 162 }, { 8, 255 }, { 1, 246 }, { 1, 175 }, { 1, 95 }, { 1, 19 }, { 66, 0 }, { 1, 90 }, { 6, 255 }, { 1, 237 }, { 1, 160 }, { 1, 80 }, { 1, 10 }, { 68, 0 }, { 1, 27 }, { 1, 241 }, { 3, 255 }, { 1, 225 }, { 1, 145 }, { 1, 65 }, { 1, 4 }, { 71, 0 }, { 1, 126 }, { 1, 255 }, { 1, 211 }, { 1, 130 }, { 1, 50 }, { 75, 0 }, { 1, 7 }, { 1, 19 }, { 2135, 0 } }, 
	LogoRLEFrame{ { 1315, 0 }, { 1, 8 }, { 78, 0 }, { 1, 10 }, { 1, 13 }, { 77, 0 }, { 1, 1 }, { 1, 38 }, { 78, 0 }, { 1, 32 }, { 1, 22 }, { 77, 0 }, { 1, 13 }, { 1, 55 }, { 1, 1 }, { 77, 0 }, { 1, 52 }, { 1, 27 }, { 77, 0 }, { 1, 28 }, { 1, 53 }, { 1, 1 }, { 76, 0 }, { 1, 5 }, { 1, 59 }, { 1, 19 }, { 9, 0 }, { 1, 4 }, { 1, 141 }, { 1, 147 }, { 65, 0 }, { 1, 39 }, { 1, 47 }, { 10, 0 }, { 1, 153 }, { 1, 255 }, { 1, 224 }, { 64, 0 }, { 1, 14 }, { 1, 62 }, { 1, 12 }, { 9, 0 }, { 1, 86 }, { 2, 255 }, { 1, 232 }, { 64, 0 }, { 1, 48 }, { 1, 38 }, { 9, 0 }, { 1, 32 }, { 1, 240 }, { 2, 255 }, { 1, 212 }, { 63, 0 }, { 1, 15 }, { 1, 56 }, { 1, 3 }, { 8, 0 }, { 1, 4 }, { 1, 199 }, { 3, 255 }, { 1, 179 }, { 63, 0 }, { 1, 45 }, { 1, 17 }, { 9, 0 }, { 1, 135 }, { 4, 255 }, { 1, 83 }, { 16, 0 }, { 2, 9 }, { 44, 0 }, { 1, 12 }, { 1, 37 }, { 9, 0 }, { 1, 67 }, { 1, 253 }, { 3, 255 }, { 1, 182 }, { 15, 0 }, { 1, 2 }, { 1, 31 }, { 1, 18 }, { 45, 0 }, { 1, 29 }, { 1, 1 }, { 8, 0 }, { 1, 21 }, { 1, 230 }, { 3, 255 }, { 1, 201 }, { 1, 11 }, { 14, 0 }, { 1, 20 }, { 1, 51 }, { 1, 14 }, { 45, 0 }, { 1, 5 }, { 1, 7 }, { 8, 0 }, { 1, 1 }, { 1, 182 }, { 3, 255 }, { 1, 217 }, { 1, 20 }, { 13, 0 }, { 1, 4 }, { 1, 43 }, { 1, 55 }, { 1, 10 }, { 56, 0 }, { 1, 114 }, { 3, 255 }, { 1, 230 }, { 1, 32 }, { 13, 0 }, { 1, 17 }, { 1, 56 }, { 1, 43 }, { 1, 4 }, { 56, 0 }, { 1, 51 }, { 1, 249 }, { 2, 255 }, { 1, 241 }, { 1, 47 }, { 1, 0 }, { 1, 27 }, { 1, 167 }, { 1, 230 }, { 1, 169 }, { 1, 2 }, { 6, 0 }, { 1, 1 }, { 1, 35 }, { 1, 61 }, { 1, 27 }, { 57, 0 }, { 1, 12 }, { 1, 218 }, { 2, 255 }, { 1, 248 }, { 1, 64 }, { 1, 0 }, { 1, 46 }, { 1, 230 }, { 3, 255 }, { 1, 23 }, { 5, 0 }, { 1, 10 }, { 1, 52 }, { 1, 54 }, { 1, 12 }, { 58, 0 }, { 1, 162 }, { 2, 255 }, { 1, 253 }, { 1, 85 }, { 1, 0 }, { 1, 62 }, { 1, 240 }, { 4, 255 }, { 1, 48 }, { 4, 0 }, { 1, 16 }, { 1, 58 }, { 1, 38 }, { 1, 3 }, { 58, 0 }, { 1, 93 }, { 3, 255 }, { 1, 108 }, { 1, 0 }, { 1, 79 }, { 1, 247 }, { 5, 255 }, { 1, 73 }, { 3, 0 }, { 1, 24 }, { 1, 47 }, { 1, 12 }, { 59, 0 }, { 1, 36 }, { 1, 243 }, { 2, 255 }, { 1, 133 }, { 1, 0 }, { 1, 100 }, { 1, 252 }, { 6, 255 }, { 1, 98 }, { 2, 0 }, { 1, 26 }, { 1, 22 }, { 60, 0 }, { 1, 6 }, { 1, 204 }, { 2, 255 }, { 1, 158 }, { 1, 0 }, { 1, 123 }, { 8, 255 }, { 1, 123 }, { 1, 0 }, { 1, 11 }, { 1, 2 }, { 61, 0 }, { 1, 141 }, { 2, 255 }, { 1, 180 }, { 1, 6 }, { 1, 146 }, { 9, 255 }, { 1, 148 }, { 63, 0 }, { 1, 73 }, { 2, 255 }, { 1, 199 }, { 1, 16 }, { 1, 167 }, { 10, 255 }, { 1, 174 }, { 62, 0 }, { 1, 24 }, { 1, 234 }, { 1, 255 }, { 1, 215 }, { 1, 31 }, { 1, 186 }, { 11, 255 }, { 1, 199 }, { 61, 0 }, { 1, 1 }, { 1, 187 }, { 1, 255 }, { 1, 229 }, { 1, 52 }, { 1, 203 }, { 12, 255 }, { 1, 223 }, { 61, 0 }, { 1, 120 }, { 2, 255 }, { 1, 138 }, { 1, 217 }, { 13, 255 }, { 1, 225 }, { 60, 0 }, { 1, 56 }, { 1, 251 }, { 17, 255 }, { 1, 111 }, { 59, 0 }, { 1, 15 }, { 1, 222 }, { 16, 255 }, { 1, 253 }, { 1, 122 }, { 60, 0 }, { 1, 168 }, { 15, 255 }, { 1, 247 }, { 1, 168 }, { 1, 47 }, { 60, 0 }, { 1, 99 }, { 13, 255 }, { 1, 240 }, { 1, 164 }, { 1, 83 }, { 1, 11 }, { 61, 0 }, { 1, 40 }, { 1, 245 }, { 10, 255 }, { 1, 230 }, { 1, 151 }, { 1, 70 }, { 1, 5 }, { 63, 0 }, { 1, 8 }, { 1, 209 }, { 8, 255 }, { 1, 219 }, { 1, 138 }, { 1, 57 }, { 1, 1 }, { 66, 0 }, { 1, 147 }, { 6, 255 }, { 1, 206 }, { 1, 125 }, { 1, 44 }, { 69, 0 }, { 1, 73 }, { 3, 255 }, { 1, 252 }, { 1, 193 }, { 1, 112 }, { 1, 32 }, { 72, 0 }, { 1, 190 }, { 1, 247 }, { 1, 180 }, { 1, 99 }, { 1, 21 }, { 75, 0 }, { 1, 13 }, { 1, 5 }, { 664, 0 }, { 1, 7 }, { 1, 1 }, { 76, 0 }, { 1, 11 }, { 1, 16 }, { 1, 1 }, { 75, 0 }, { 1, 10 }, { 1, 25 }, { 1, 14 }, { 75, 0 }, { 1, 5 }, { 1, 23 }, { 1, 27 }, { 1, 7 }, { 74, 0 }, { 1, 1 }, { 1, 16 }, { 1, 30 }, { 1, 18 }, { 1, 1 }, { 74, 0 }, { 1, 9 }, { 1, 27 }, { 1, 25 }, { 1, 7 }, { 74, 0 }, { 1, 1 }, { 1, 19 }, { 1, 30 }, { 1, 15 }, { 75, 0 }, { 1, 5 }, { 1, 25 }, { 1, 17 }, { 1, 3 }, { 75, 0 }, { 1, 11 }, { 1, 16 }, { 1, 3 }, { 76, 0 }, { 1, 6 }, { 1, 2 }, { 764, 0 } }, 

} };
	
} // namespace RoundVideoData
