/*
 * Copyright 2018-2025 Gentoo Foundation
 * Distributed under the terms of the GNU General Public License v2
 *
 * Copyright 2018-     Fabian Groffen  - <grobian@gentoo.org>
 */

#ifndef _HASH_H
#define _HASH_H 1

/* for AT_FDCWD */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifdef HAVE_BLAKE2B
# include <blake2.h>
#endif

#include "md5.h"
#include "sha1.h"
#include "sha256.h"
#include "sha512.h"

#define BLAKE2B_DIGEST_SIZE    BLAKE2B_OUTBYTES

#define MD5_DIGEST_LENGTH      (MD5_DIGEST_SIZE * 2)
#define SHA1_DIGEST_LENGTH     (SHA1_DIGEST_SIZE * 2)
#define SHA256_DIGEST_LENGTH   (SHA256_DIGEST_SIZE * 2)
#define SHA512_DIGEST_LENGTH   (SHA512_DIGEST_SIZE * 2)
#define BLAKE2B_DIGEST_LENGTH  (BLAKE2B_DIGEST_SIZE * 2)

enum hash_impls {
	HASH_MD5       = 1<<0,
	HASH_SHA1      = 1<<1,
	HASH_SHA256    = 1<<2,
	HASH_SHA512    = 1<<3,
	HASH_WHIRLPOOL = 1<<4,  /* removed */
	HASH_BLAKE2B   = 1<<5
};

/* default changed from sha256, sha512, whirlpool
 * to blake2b, sha512 on 2017-11-21 */
#define HASH_DEFAULT  (HASH_BLAKE2B | HASH_SHA512);

/* pass in a fd and get back a fd; filename is for display only */
typedef int (*hash_cb_t) (int, const char *);

void hash_hex(char *out, const unsigned char *buf, const int length);
int hash_multiple_file_fd(
		int fd, char *md5, char *sha1, char *sha256, char *sha512,
		char *blak2b, size_t *flen, int hashes);
int hash_multiple_file_at_cb(
		int pfd, const char *fname, hash_cb_t cb, char *md5,
		char *sha1, char *sha256, char *sha512,
		char *blak2b, size_t *flen, int hashes);
#define hash_multiple_file(f, m, s1, s2, s5, b, l, h) \
	hash_multiple_file_at_cb(AT_FDCWD, f, NULL, m, s1, s2, s5, b, l, h)
#define hash_compute_file(f, s2, s5, b, l, h) \
	hash_multiple_file_at_cb(AT_FDCWD, f, NULL, NULL, NULL, s2, s5, b, l, h)
char *hash_file_at_cb(int pfd, const char *filename, int hash_algo, hash_cb_t cb);
#define hash_file(f, h) hash_file_at_cb(AT_FDCWD, f, h, NULL)
#define hash_file_at(fd, f, h) hash_file_at_cb(fd, f, h, NULL)
char *hash_string(const char *buf, ssize_t buflen, int hash);

#endif
