(*:Name: NonConvexTriangulate` *)

(*:Title: NonConvexTriangulate *)

(*:Author: Tom Wickham-Jones*)

(*:Package Version: 1.0 *)

(*:Mathematica Version: 2.2 *)

(*:Summary:
	This package returns a non-convex triangulation of a given
	three-dimensional polygon.
*)

(*:History:
	Created summer 1993 by Tom Wickham-Jones.

	This package is described in the book
	Mathematica Graphics: Techniques and Applications.
	Tom Wickham-Jones, TELOS/Springer-Verlag 1994.

*)


BeginPackage["ExtendGraphics`NonConvexTriangulate`"]

NonConvexTriangulate::usage =
	"NonConvexTriangulate[ pts] returns a triangulation
	of the points pts that maintains the outline."
	
	
NonConvexTriangulate::bdpts = 
"An error has occured in NonConvexTriangulate."

NonConvexTriangulate::collinear = 
"Collinear points have been found."

Begin["`Private`"]

NonConvexTriangulate[ ipts_List] :=
	Block[{clock, stpt, res, pts, flag = True, side},
		pts = N[ ipts] ;
		stpt = LeftLowVertex[ pts] ;
		clock = ClockTest[
					modPart[ pts, stpt-1],
					modPart[ pts, stpt],
					modPart[ pts, stpt+1]] ;
		If[ clock === Collinear,
			Return[ BadTriang[pts]]] ;
		side = 1 ;
		While[ flag,
			res = ReducePoints[ pts, Range[ Length[ pts]], clock, 0, side] ;
			If[ MemberQ[ res, Error],
					side = 2*side ;
					If[ side > Length[ pts],
							Return[ BadTriang[pts]]] ;
				,
					flag = False]] ;
		res
		]


BadTriang[ pts_] :=
	(
	Message[ NonConvexTriangulate::bdpts] ;
	Table[ {1, i,i+1}, {i,2,Length[ pts]-1}]
	)

	

(*
 ClockTest[ pl, p, pn]
 is the minor angle (pp-p) to (pn-p) clock-wise or not.
*)
 
 
ClockTest[ {ppx_, ppy_}, {px_, py_}, {pnx_, pny_}] :=
        Block[{test},
                test = (pnx - px)*(ppy - py) +
                                (-ppx + px)*(pny - py) ;
                Which[ Positive[ test],
                           False,
                       Negative[ test],
                           True,
                       True,
							Collinear]
                ]

(*
	Part with mod index.
	modPart[ {a,b,c}, 4] -> a

	modPart[ {a,b,c}, 0] -> c
*)

modPart[ pts_, n_] := 
		Part[ pts, Mod[ n-1, Length[ pts]]+1]


(*
 Working with the first of pts.  
 	Is the sense equal to clock?
		if not then reject this point.
		if true then test to see if any of these points
				fall inside the rest.
*)

NormalVector[ {px_, py_}] :=
    {py, -px}

VectorLength[ p_] :=
	Sqrt[ p.p]

SideOfLine[ p_, {p1_, p2_}] :=
	Block[{u, v, test},
		u = (p2 - p1) ;
		u = u/VectorLength[ u] ;
		v = (p - p1) ;
		v = v/VectorLength[ v] ;
		test = u.NormalVector[v] ;
		If[ test > 10^(-15.),
			Left,
			If[ test < -10^(-15.),
				Right,
				Collinear]]
		]
		
		

PtInTri[ p_, {p1_, p2_, p3_}] :=
	Block[{test}, 
		If[ p == p1 || p == p2 || p == p3,
			False,
			test = 
		  		Map[ SideOfLine[ p,#]&, {{p1,p2},{p2,p3},{p3,p1}}] ;
			If[ MemberQ[ test, Collinear],
					test = DeleteCases[ test, Collinear] ;
					If[ Apply[ SameQ, test],
						Collinear,
						False]
					,
					Apply[ SameQ, test]]
			]
		]



ClockTriTest[ pts_, indices_, clock_, side_] :=
	Block[{p1, p2, p3, test, rest},
		{p1, p2, p3} = Part[ pts, Take[ indices, 3]] ;
		test = ClockTest[ p1, p2, p3] ;
		Which[
			test === Collinear, Error,
			test =!= clock, False,
			True, 
				test = Drop[ indices, 3] ;
				If[ 2*side < Length[ test],
					test = Join[ Take[ test, side], Take[ test, -side]]] ;
				rest = Part[ pts, test] ;
				rest = Map[ Not[ 
						PtInTri[ #, {p1, p2, p3}]]&, rest] ;
				Apply[ And, rest] === True]
	]



ReducePoints[ pts_, indices_ /; Length[ indices] > 3, clock_, count_, side_] :=
	If[ ClockTriTest[ pts, indices, clock, side],
                Prepend[
                   ReducePoints[ pts, Drop[ indices, {2}], clock, 0, side],
				   Take[ indices, 3]],
				If[ count > Length[ indices], 
					{Error},
					ReducePoints[ pts, RotateLeft[ indices], clock, count+1, side]],
				{Error}]
		

ReducePoints[ pts_, indices_, clock_, count_, side_] :=
        If[ Apply[ ClockTest, Part[ pts, indices]] =!= clock,
		        {Error},
				{indices}]


(* Return True if p1 < p2 *)
CompareFun[ {p1x_, p1y_}, {p2x_, p2y_}] :=
        If[ p1x === p2x,
                        p1y < p2y,
                        p1x < p2x]


(* LeftLowVertex finds the left-most (lowest) vertex *)
(* This is necessary to make sure it is a convex vertex *)

LeftLowVertex[ pts_List] :=
        Block[{iwant, want},
                iwant = 1;
                want = First[ pts] ;
                Do[
                   If[ CompareFun[ Part[ pts, i], want],
                       iwant = i; want = Part[ pts, i]],
                   {i,2,Length[ pts]}] ;
                iwant
                ]    

End[]

EndPackage[]

 
(*:Examples:

<<ExtendGraphics`NonConvexTriangulate`

pt = {{0, 0}, {1, -2}, {1.5, -2}, {4, 4}, {2.5, 4}};

NonConvexTriangulate[ pt]

Show[ 
  Graphics[ 
    Map[ Line[ Part[ pt, #] /. {a_,b__} -> {a,b,a}]&, %]]]

*)
 
    
