/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author: Pawel Stolowski <pawel.stolowski@canonical.com>
 */

#include "SmartScopesParser.h"
#include "Utils.h"
#include <QJsonDocument>
#include <QJsonObject>
#include <QJsonArray>
#include <deelistmodel.h>
#include <QDebug>

void SmartScopesParser::parseLine(const QString& text, const ResultHandler& resultHandler, const RecommendationsHandler &recommendationsHandler)
{
    const QJsonDocument doc = QJsonDocument::fromJson(text.toUtf8());
    const QJsonObject root = doc.object();

    auto it = root.find("type");
    if (it == root.end())
    {
        qWarning() << "Missing 'type' element";
        return;
    }

    const QString repType = it.value().toString();
    if (repType == "results")
        handleResults(root, resultHandler);
    else if (repType == "recommendations")
        handleRecommendations(root, recommendationsHandler);
    else
        qWarning() << "Unsupported reply type:" << repType;
}

void SmartScopesParser::handleResults(const QJsonObject& root, const ResultHandler& resultHandler)
{
    auto it = root.find("info");
    if (it == root.end())
    {
        qWarning() << "Missing 'info' element";
        return;
    }

    const QJsonObject infoObj = it.value().toObject();
    for (auto infoIt = infoObj.begin(); infoIt != infoObj.end(); infoIt++)
    {
        const QString scopeId = infoIt.key();
        const QJsonArray results = infoIt.value().toArray();
        for (auto resultsIt = results.begin(); resultsIt != results.end(); resultsIt++)
        {
            auto result = std::shared_ptr<SmartScopesResult>(new SmartScopesResult());

            const QJsonObject resultObj = (*resultsIt).toObject();
            it = resultObj.find("uri");
            if (it == resultObj.end())
                continue;
            result->uri = it.value().toString();

            it = resultObj.find("title");
            if (it == resultObj.end())
                continue;
            result->title = it.value().toString();

            it = resultObj.find("metadata");
            if (it == resultObj.end())
                continue;
            const auto metadata = it.value().toObject().toVariantMap();

            // convert qvariant map returned by json parser to qvariant hash;
            // this is needed because DeeListModel::DataFromVariant() doesn't support QVariantMap.
            result->metadata = QVariantMapToHash(metadata);

            it = resultObj.find("icon_hint");
            if (it == resultObj.end())
                continue;
            result->icon_hint = it.value().toString();

            it = resultObj.find("mimetype");
            if (it == resultObj.end())
                result->mimetype = "text/html";
            else
                result->mimetype = it.value().toString();

            it = resultObj.find("comment");
            if (it != resultObj.end())
                result->comment = it.value().toString();

            it = resultObj.find("dnd_uri");
            if (it != resultObj.end())
                result->dnd_uri = it.value().toString();
            else
                result->dnd_uri = "";

            resultHandler(scopeId, result);
        }
    }
}

void SmartScopesParser::handleRecommendations(const QJsonObject& root, const RecommendationsHandler& recommendationsHandler)
{
    auto it = root.find("server_sid");
    if (it == root.end())
    {
        qWarning() << "Missing 'server_sid' element";
        return;
    }

    auto server_sid = it.value().toString();
    recommendationsHandler(server_sid);
}

GVariant* SmartScopesResult::toGVariant() const
{
    auto metavar = DeeListModel::DataFromVariant(metadata);

    GVariant* children[] = {
        g_variant_new_string(uri.toUtf8().constData()),
        g_variant_new_string(icon_hint.toUtf8().constData()),
        g_variant_new_uint32(category),
        g_variant_new_uint32(resultType),
        g_variant_new_string(mimetype.toUtf8().constData()),
        g_variant_new_string(title.toUtf8().constData()),
        g_variant_new_string(comment.toUtf8().constData()),
        g_variant_new_string(dnd_uri.toUtf8().constData()),
        metavar
    };

    auto array = g_variant_new_tuple(children, sizeof(children)/sizeof(GVariant*));
    return array;
}
