/*
 * Copyright (C) 2024 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "JSIteratorHelper.h"

#include "JSCInlines.h"
#include "JSInternalFieldObjectImplInlines.h"

namespace JSC {

const ClassInfo JSIteratorHelper::s_info = { "Iterator Helper"_s, &Base::s_info, nullptr, nullptr, CREATE_METHOD_TABLE(JSIteratorHelper) };

void JSIteratorHelper::finishCreation(VM& vm, JSValue generator, JSValue underlyingIterator)
{
    Base::finishCreation(vm);
    internalField(Field::Generator).set(vm, this, generator);
    internalField(Field::UnderlyingIterator).set(vm, this, underlyingIterator);
}

JSIteratorHelper* JSIteratorHelper::createWithInitialValues(VM& vm, Structure* structure)
{
    auto values = initialValues();
    JSIteratorHelper* result = new (NotNull, allocateCell<JSIteratorHelper>(vm)) JSIteratorHelper(vm, structure);
    result->finishCreation(vm, values[0], values[1]);
    return result;
}

JSIteratorHelper* JSIteratorHelper::create(VM& vm, Structure* structure, JSValue generator, JSValue underlyingIterator)
{
    ASSERT(generator.isObject() && (underlyingIterator.isObject() || underlyingIterator.isNull()));
    JSIteratorHelper* result = new (NotNull, allocateCell<JSIteratorHelper>(vm)) JSIteratorHelper(vm, structure);
    result->finishCreation(vm, generator, underlyingIterator);
    return result;
}

Structure* JSIteratorHelper::createStructure(VM& vm, JSGlobalObject* globalObject, JSValue prototype)
{
    return Structure::create(vm, globalObject, prototype, TypeInfo(JSIteratorHelperType, StructureFlags), info());
}

JSIteratorHelper::JSIteratorHelper(VM& vm, Structure* structure)
    : Base(vm, structure)
{
}

template<typename Visitor>
void JSIteratorHelper::visitChildrenImpl(JSCell* cell, Visitor& visitor)
{
    auto* thisObject = jsCast<JSIteratorHelper*>(cell);
    ASSERT_GC_OBJECT_INHERITS(thisObject, info());
    Base::visitChildren(thisObject, visitor);
}

DEFINE_VISIT_CHILDREN(JSIteratorHelper);

JSC_DEFINE_HOST_FUNCTION(iteratorHelperPrivateFuncCreate, (JSGlobalObject* globalObject, CallFrame* callFrame))
{
    return JSValue::encode(JSIteratorHelper::create(globalObject->vm(), globalObject->iteratorHelperStructure(), callFrame->uncheckedArgument(0), callFrame->uncheckedArgument(1)));
}

} // namespace JSC
